<?php
/**
 * Plugin Name: SVG
 * Plugin URI: https://plugins.kddev.co.uk/kdweb-svg/
 * Description: Enable support for .svg (Scalable Vector Graphics) files in the WordPress media library with sanitization.
 * Version: 1.0.0
 * Author: KD Web
 * Author URI: https://www.kddev.co.uk
 * Requires at least: 6.0.0
 * Requires PHP: 8.0.0
 *
 * @package KDWeb\Plugin\SVG
 */

namespace KDWeb\Plugin\SVG;

require_once __DIR__ . '/vendor/autoload.php';

use enshrined\svgSanitize\data\AllowedTags;
use enshrined\svgSanitize\data\AllowedAttributes;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

add_filter( 'upload_mimes', 'KDWeb\Plugin\SVG\svg_support_upload_mimes', 99 );
add_filter( 'wp_check_filetype_and_ext', 'KDWeb\Plugin\SVG\svg_support_check_mime_on_upload', 10, 4 );
add_action( 'admin_init', 'KDWeb\Plugin\SVG\svg_support_admin_fix_media_list_thumbs_start' );
add_filter( 'final_output', 'KDWeb\Plugin\SVG\svg_support_admin_fix_media_list_thumbs_end' );
add_filter( 'wp_prepare_attachment_for_js', 'KDWeb\Plugin\SVG\svg_support_prepare_attachment_for_js', 10, 2 );
add_filter( 'wp_generate_attachment_metadata', 'KDWeb\Plugin\SVG\svg_support_generate_attachment_meta', 10, 3 );
add_filter( 'wp_handle_upload_prefilter', 'KDWeb\Plugin\SVG\svg_support_handle_upload_prefilter' );
add_filter( 'wp_calculate_image_srcset', 'KDWeb\Plugin\SVG\svg_support_disable_srcset' );
add_filter( 'wp_get_attachment_image_src', 'KDWeb\Plugin\SVG\svg_support_dimension_fix', 10, 2 );
add_action( 'admin_enqueue_scripts', 'KDWeb\Plugin\SVG\svg_support_admin_assets' );
add_filter( 'wp_kses_allowed_html', 'KDWeb\Plugin\SVG\extend_kses_rules_with_svg', 10, 2 );

/**
 * Add svg to allowed mime types for uploading to the media library
 *
 * @param array $mimes Mime types.
 * @return array
 */
function svg_support_upload_mimes( $mimes = array() ) {
	if ( ! apply_filters( 'kdweb_user_can_upload_svg', true, get_current_user() ) ) {
		return $mimes;
	}
	$mimes['svg']  = 'image/svg+xml';
	$mimes['svgz'] = 'image/svg+xml';
	return $mimes;
}

/**
 * Apply correct mime type checks for SVG files upon upload
 *
 * @param array  $checked The check array.
 * @param string $file The filepath.
 * @param string $filename The name of the file.
 * @param array  $mimes Mime types.
 * @return array
 */
function svg_support_check_mime_on_upload( $checked, $file, $filename, $mimes ) {
	if ( $checked['type'] ) {
		return $checked;
	}
	$filetype = wp_check_filetype( $filename, $mimes );
	if ( $filetype['type'] && 0 === strpos( $filetype['type'], 'image/' ) && 'svg' !== $filetype['ext'] ) {
		$filetype['ext']  = false;
		$filetype['type'] = false;
	}
	return array(
		'ext'             => $filetype['ext'],
		'type'            => $filetype['type'],
		'proper_filename' => $filename,
	);
}

/**
 * Add ability to view thumbnails that are svg files (start html filter)
 *
 * @return void
 */
function svg_support_admin_fix_media_list_thumbs_start() {
	$screen = get_current_screen();
	if ( ! is_object( $screen ) || 'upload' !== $screen->id ) {
		return;
	}
	ob_start( fn( $content ) => apply_filters( 'final_output', $content ) );
}

/**
 * Add ability to view thumbnails that are svg files (end html filter)
 *
 * @param string $content The entire html content.
 * @return string
 */
function svg_support_admin_fix_media_list_thumbs_end( $content ) {
	$screen = get_current_screen();
	if ( ! is_object( $screen ) || 'upload' !== $screen->id ) {
		return $content;
	}
	return str_replace(
		array(
			'<# } else if ( \'image\' === data.type && data.sizes && data.sizes.full ) { #>',
			'<# } else if ( \'image\' === data.type && data.sizes ) { #>',
		),
		array(
			'<# } else if ( \'svg+xml\' === data.subtype ) { #>
				<img class="details-image" src="{{ data.url }}" draggable="false" />
				<# } else if ( \'image\' === data.type && data.sizes && data.sizes.full ) { #>',
			'<# } else if ( \'svg+xml\' === data.subtype ) { #>
				<div class="centered">
					<img src="{{ data.url }}" class="thumbnail" draggable="false" />
				</div>
				<# } else if ( \'image\' === data.type && data.sizes ) { #>',
		),
		$content
	);
}

/**
 * Prepare svg width and height for js response
 *
 * @param array  $response The response settings.
 * @param object $attachment The attachment data.
 * @return array
 */
function svg_support_prepare_attachment_for_js( $response, $attachment ) {
	if ( 'image/svg+xml' !== $response['mime'] || ! empty( $response['sizes'] ) ) {
		return $response;
	}
	$path = get_attached_file( $attachment->ID );
	if ( ! file_exists( $path ) ) {
		$path = $response['url'];
	}
	$dimensions        = get_svg_file_dimensions( $path );
	$response['sizes'] = array(
		'full' => array(
			'url'         => $response['url'],
			'width'       => $dimensions['width'],
			'height'      => $dimensions['height'],
			'orientation' => $dimensions['width'] > $dimensions['height'] ? 'landscape' : 'portrait',
		),
	);
	return $response;
}

/**
 * Generate metadata for svg files.
 *
 * @param array $meta The meta data array.
 * @param int   $attachment_id The attachment ID.
 * @return array
 */
function svg_support_generate_attachment_meta( $meta, $attachment_id ) {
	global $_wp_additional_image_sizes;
	$mime = get_post_mime_type( $attachment_id );
	if ( 'image/svg+xml' !== $mime ) {
		return $meta;
	}
	$path       = get_attached_file( $attachment_id );
	$filename   = basename( $path );
	$dimensions = get_svg_file_dimensions( $path );
	$meta       = array(
		'width'  => $dimensions['width'],
		'height' => $dimensions['height'],
		'file'   => $filename,
	);
	$sizes      = array();
	foreach ( get_intermediate_image_sizes() as $size ) {
		$width  = $dimensions['width'];
		$height = $dimensions['height'];
		$crop   = false;
		if ( 0 !== $width && 0 !== $height ) {
			$width  = isset( $_wp_additional_image_sizes[ $size ]['width'] ) ? intval( $_wp_additional_image_sizes[ $size ]['width'] ) : get_option( "{$size}_size_w" );
			$ratio  = round( $width > $height ? $width / $height : $height / $width, 2 );
			$height = round( $width > $height ? $width / $ratio : $width * $ratio );
		} else {
			$width  = isset( $_wp_additional_image_sizes[ $size ]['width'] ) ? intval( $_wp_additional_image_sizes[ $size ]['width'] ) : get_option( "{$size}_size_w" );
			$height = isset( $_wp_additional_image_sizes[ $size ]['height'] ) ? intval( $_wp_additional_image_sizes[ $size ]['height'] ) : get_option( "{$size}_size_h" );
			$crop   = isset( $_wp_additional_image_sizes[ $size ]['crop'] ) ? intval( $_wp_additional_image_sizes[ $size ]['crop'] ) : get_option( "{$size}_crop" );
		}
		$sizes[ $size ] = array(
			'file'      => $filename,
			'width'     => $width,
			'height'    => $height,
			'crop'      => $crop,
			'mime-type' => 'image/svg+xml',
		);
	}
	$meta['sizes'] = $sizes;
	return $meta;
}

/**
 * Sanitize SVG file before uploading to the media library.
 *
 * @param string $file The file array.
 * @return array
 */
function svg_support_handle_upload_prefilter( $file ) {
	if ( ! isset( $file['type'] ) || 'image/svg+xml' !== $file['type'] ) {
		return $file;
	}
	$sanitized_file = sanitize_svg_file( $file['tmp_name'] );
	if ( ! $sanitized_file ) {
		$file['error'] = __( 'Error sanitizing SVG file. The file may be invalid.', 'kdweb' );
	}
	return $file;
}

/**
 * Disable srcset for svg files
 *
 * @param array $sources URL sources for the attachment.
 * @return array
 */
function svg_support_disable_srcset( $sources ) {
	$first_source = reset( $sources );
	if ( empty( $first_source['url'] ) ) {
		return $sources;
	}
	$extension = pathinfo( $first_source['url'], PATHINFO_EXTENSION );
	if ( 'svg' !== $extension ) {
		return $sources;
	}
	return array();
}

/**
 * Fix svg width and height so that there is no division with zero
 *
 * @param array $image The image array.
 * @param int   $attachment_id The attachment ID.
 * @return array
 */
function svg_support_dimension_fix( $image, $attachment_id ) {
	if ( 'image/svg+xml' !== get_post_mime_type( $attachment_id ) ) {
		return $image;
	}
	if ( isset( $image[1] ) && 0 === $image[1] ) {
		$image[1] = 1;
	}
	if ( isset( $image[2] ) && 0 === $image[2] ) {
		$image[2] = 1;
	}
	return $image;
}

/**
 * Register SVG support admin assets
 *
 * @return void
 */
function svg_support_admin_assets() {
	global $pagenow;
	$screen = get_current_screen();
	if ( ! is_object( $screen ) || ! is_admin() ) {
		return;
	}
	if ( 'upload' === $screen->id ) {
		wp_enqueue_style( 'kdweb-svg-support-admin', plugin_dir_url( __FILE__ ) . 'assets/media-list-icon.css', array(), '1.0.0' );
	}
	if ( in_array( $pagenow, array( 'post.php', 'post-new.php' ), true ) ) {
		wp_enqueue_style( 'kdweb-svg-support-admin', plugin_dir_url( __FILE__ ) . 'assets/admin-editor.css', array(), '1.0.0' );
	}
}

/**
 * Extend the escaped, allowed HTML tags and attributes for SVG
 *
 * @param array  $html The array of html elements and their attributes that are allowed.
 * @param string $context The kses context.
 * @return array
 */
function extend_kses_rules_with_svg( $html, $context ) {
	if ( 'post' !== $context ) {
		return $html;
	}
	$attributes             = array_fill_keys( ( new SvgSanitizerAllowedAttributes() )->getAttributes(), true );
	$attributes['aria-*']   = true;
	$attributes['data-*']   = true;
	foreach ( ( new SvgSanitizerAllowedTags() )->getTags() as $tag ) {
		$html[ $tag ] = $attributes;
	}
	return $html;
}

/**
 * Retrieve an SVG file's dimensions
 *
 * @param string $file The full path of an SVG file.
 * @return array
 */
function get_svg_file_dimensions( $file ) {
	$width  = 0;
	$height = 0;
	$svg    = file_exists( $file ) ? simplexml_load_file( $file ) : false;
	if ( false !== $svg ) {
		$attributes = $svg->attributes();
		$viewbox    = explode( ' ', $attributes->viewBox );
		$width      = $attributes->width ?? $viewbox[2] ?? 0;
		$height     = $attributes->height ?? $viewbox[3] ?? 0;
	}
	return array(
		'width'  => intval( $width ),
		'height' => intval( $height ),
	);
}

/**
 * Sanitize SVG file
 *
 * @param string  $svg Path to the svg file.
 * @param boolean $minify Whether to minify or not.
 * @return boolean
 */
function sanitize_svg_file( $svg, $minify = true ) {
	if ( ! file_exists( $svg ) ) {
		return false;
	}
	return (bool) file_put_contents(
		$svg,
		sanitize_svg(
			file_get_contents( $svg ),
			$minify
		)
	);
}

/**
 * Sanitize SVG string
 *
 * @param string  $svg The SVG string to sanitize.
 * @param boolean $minify Whether to minify or not.
 * @return string
 */
function sanitize_svg( $svg, $minify = true ) {
	global $_kdweb_svg_sanitizer;
	if ( ! isset( $_kdweb_svg_sanitizer ) ) {
		$_kdweb_svg_sanitizer = new \enshrined\svgSanitize\Sanitizer();
		$_kdweb_svg_sanitizer->setAllowedTags( new SvgSanitizerAllowedTags() );
		$_kdweb_svg_sanitizer->setAllowedAttrs( new SvgSanitizerAllowedAttributes() );
		$_kdweb_svg_sanitizer->minify( $minify );
	}
	return $_kdweb_svg_sanitizer->sanitize( $svg );
}

/**
 * Sanitize SVG with filterable tags
 */
class SvgSanitizerAllowedTags extends AllowedTags {

	/**
	 * Returns an array of tags
	 *
	 * @return array
	 */
	public static function getTags() {
		return apply_filters( 'kdweb_svg_allowed_tags', parent::getTags() );
	}
}

/**
 * Sanitize SVG with filterable attributes
 */
class SvgSanitizerAllowedAttributes extends AllowedAttributes {

	/**
	 * Returns an array of attributes
	 *
	 * @return array
	 */
	public static function getAttributes() {
		return apply_filters( 'kdweb_svg_allowed_attributes', parent::getAttributes() );
	}
}

/**
 * Check for plugin updates.
 *
 * @param object $transient The transient object.
 * @return object The updated transient object.
 */
function check_plugin_update( $transient ) {
	$current_version = get_file_data( __FILE__, array( 'Version' => 'Version' ) )['Version'];
	$plugin_info = json_decode(
		wp_remote_retrieve_body(
			wp_remote_get( 'https://plugins.kddev.co.uk/kdweb-svg/update' )
		),
		true
	);
	if ( $plugin_info && ! is_wp_error( $plugin_info ) && version_compare( $plugin_info['new_version'], $current_version, '>' ) ) {
		$transient->response['kdweb-svg/kdweb-svg.php'] = (object) $plugin_info;
	}
	$transient->no_update['kdweb-svg/kdweb-svg.php'] = (object) array(
		'id'            => 'kdweb-svg/kdweb-svg.php',
		'slug'          => 'kdweb-svg',
		'plugin'        => 'kdweb-svg/kdweb-svg.php',
		'new_version'   => get_file_data( __FILE__, array( 'Version' => 'Version' ) )['Version'],
		'url'           => '',
		'package'       => '',
		'icons'         => array(),
		'banners'       => array(),
		'banners_rtl'   => array(),
		'tested'        => '',
		'requires_php'  => '',
		'compatibility' => new \stdClass(),
	);
	return $transient;
}
add_filter( 'pre_set_site_transient_update_plugins', 'KDWeb\Plugin\SVG\check_plugin_update' );

/**
 * Get plugin information from the plugin server.
 *
 * @param bool   $result The default value.
 * @param string $action The action to perform.
 * @param object $args The arguments for the action.
 * @return object|bool The plugin information or false.
 */
function plugin_info( $result, $action, $args ) {
	if ( 'plugin_information' === $action && 'kdweb-svg' === $args->slug ) {
		$plugin_info = json_decode(
			wp_remote_retrieve_body(
				wp_remote_get( 'https://plugins.kddev.co.uk/kdweb-svg/info' )
			),
			true
		);
		if ( ! $plugin_info || is_wp_error( $plugin_info ) ) {
			return $result;
		}
		return (object) $plugin_info;
	}
	return $result;
}
add_filter( 'plugins_api', 'KDWeb\Plugin\SVG\plugin_info', 10, 3 );
