<?php
/**
 * Plugin Name: Rive
 * Plugin URI: https://plugins.kddev.co.uk/kdweb-rive/
 * Description: Enable support for .riv (Rive) files in the WordPress media library.
 * Version: 0.0.1
 * Author: KD Web
 * Author URI: https://www.kddev.co.uk
 * Requires at least: 6.0.0
 * Requires PHP: 8.0.0
 *
 * @package KDWeb\Plugin\Rive
 */

namespace KDWeb\Plugin\Rive;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Add svg to allowed mime types for uploading to the media library
 *
 * @param array $mimes Mime types.
 * @return array
 */
function rive_support_upload_mimes( $mimes = array() ) {
	if ( ! apply_filters( 'kdweb_user_can_upload_rive', true, get_current_user() ) ) {
		return $mimes;
	}
	$mimes['riv'] = 'application/octet-stream';
	return $mimes;
}
add_filter( 'upload_mimes', 'KDWeb\Plugin\Rive\rive_support_upload_mimes', 99 );

/**
 * Extend the escaped, allowed HTML tags and attributes for rive canvases
 *
 * @param array $html The array of html elements and their attributes that are allowed.
 * @param string $context The kses context.
 * @return array
 */
function extend_kses_rules_with_rive( $html, $context ) {
	if ( 'post' !== $context ) {
        return $html;
    }
	$html['canvas'] = array(
		'aria-*' => true,
		'class' => true,
		'id'    => true,
		'data-*' => true,
		'height' => true,
		'hidden' => true,
		'width' => true,
		'style' => true,
		'xml:lang' => true,
	);
    return $html;
}
add_filter( 'wp_kses_allowed_html', 'KDWeb\Plugin\Rive\extend_kses_rules_with_rive', 10, 2 );

/**
 * Check for plugin updates.
 *
 * @param object $transient The transient object.
 * @return object The updated transient object.
 */
function check_plugin_update( $transient ) {
	$current_version = get_file_data( __FILE__, array( 'Version' => 'Version' ) )['Version'];
	$plugin_info = json_decode(
		wp_remote_retrieve_body(
			wp_remote_get( 'https://plugins.kddev.co.uk/kdweb-rive/update' )
		),
		true
	);
	if ( $plugin_info && ! is_wp_error( $plugin_info ) && version_compare( $plugin_info['new_version'], $current_version, '>' ) ) {
		$transient->response['kdweb-rive/kdweb-rive.php'] = (object) $plugin_info;
	}
	$transient->no_update['kdweb-rive/kdweb-rive.php'] = (object) array(
		'id'            => 'kdweb-rive/kdweb-rive.php',
		'slug'          => 'kdweb-rive',
		'plugin'        => 'kdweb-rive/kdweb-rive.php',
		'new_version'   => get_file_data( __FILE__, array( 'Version' => 'Version' ) )['Version'],
		'url'           => '',
		'package'       => '',
		'icons'         => array(),
		'banners'       => array(),
		'banners_rtl'   => array(),
		'tested'        => '',
		'requires_php'  => '',
		'compatibility' => new \stdClass(),
	);
	return $transient;
}
add_filter( 'pre_set_site_transient_update_plugins', 'KDWeb\Plugin\Rive\check_plugin_update' );

/**
 * Get plugin information from the plugin server.
 *
 * @param bool   $result The default value.
 * @param string $action The action to perform.
 * @param object $args The arguments for the action.
 * @return object|bool The plugin information or false.
 */
function plugin_info( $result, $action, $args ) {
	if ( 'plugin_information' === $action && 'kdweb-rive' === $args->slug ) {
		$plugin_info = json_decode(
			wp_remote_retrieve_body(
				wp_remote_get( 'https://plugins.kddev.co.uk/kdweb-rive/info' )
			),
			true
		);
		if ( ! $plugin_info || is_wp_error( $plugin_info ) ) {
			return $result;
		}
		return (object) $plugin_info;
	}
	return $result;
}
add_filter( 'plugins_api', 'KDWeb\Plugin\Rive\plugin_info', 10, 3 );
