<?php
/**
 * Plugin Name: Duplicate
 * Plugin URI: https://plugins.kddev.co.uk/kdweb-duplicate/
 * Description: Add duplicate post feature to the WordPress admin for posts, pages and custom post types.
 * Version: 1.0.0
 * Author: KD Web
 * Author URI: https://www.kddev.co.uk
 * Requires at least: 6.0.0
 * Requires PHP: 8.0.0
 *
 * @package KDWeb\Plugin\Duplicate
 */

namespace KDWeb\Plugin\Duplicate;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

add_action( 'admin_action_kdweb_duplicate_post', 'KDWeb\Plugin\Duplicate\duplicate_post_action' );
add_filter( 'post_row_actions', 'KDWeb\Plugin\Duplicate\duplicate_post_to_post_row_actions', 10, 2 );
add_filter( 'page_row_actions', 'KDWeb\Plugin\Duplicate\duplicate_post_to_post_row_actions', 10, 2 );
add_action( 'admin_head', 'KDWeb\Plugin\Duplicate\duplicate_post_admin_assets' );
add_action( 'post_submitbox_misc_actions', 'KDWeb\Plugin\Duplicate\duplicate_post_to_edit_page' );

/**
 * Duplicate the post when the action is triggered
 *
 * @return void
 */
function duplicate_post_action() {
	$id      = intval( $_REQUEST['post'] );
	$nonce   = sanitize_text_field( $_REQUEST['nonce'] );
	$user_id = get_current_user_id();
	if ( ! wp_verify_nonce( $nonce, "kdweb-duplicate-{$id}" ) ) {
		wp_die( __( 'Security check failed.' ) );
	}
	$post = get_post( $id );
	if ( empty( $post ) ) {
		wp_die( __( 'Page not found' ) );
	}
	if ( ! current_user_can( get_post_type_object( $post->post_type )->cap->edit_posts ) && ! current_user_can( 'manage_options' ) ) {
		wp_die( __( 'Sorry, you are not allowed to do that.' ) );
	}
	$postdata = array(
		'comment_status'        => $post->comment_status,
		'menu_order'            => $post->menu_order,
		'ping_status'           => $post->ping_status,
		'post_author'           => $user_id,
		'post_content'          => $post->post_content,
		'post_content_filtered' => $post->post_content_filtered,
		'post_excerpt'          => $post->post_excerpt,
		'post_parent'           => $post->post_parent,
		'post_password'         => $post->post_password,
		'post_status'           => 'draft',
		'post_title'            => $post->post_title,
		'post_type'             => $post->post_type,
		'to_ping'               => $post->to_ping,
	);
	$new      = wp_insert_post( wp_slash( $postdata ), false, false );
	if ( is_wp_error( $new ) ) {
		wp_die( __( 'An error has occurred. Please reload the page and try again.' ) );
	}
	$taxonomies = get_object_taxonomies( $post );
	if ( ! empty( $taxonomies ) && is_array( $taxonomies ) ) {
		foreach ( $taxonomies as $taxonomy ) {
			$terms = wp_get_object_terms( $id, $taxonomy, array( 'fields' => 'slugs' ) );
			wp_set_object_terms( $new, $terms, $taxonomy, false );
		}
	}
	$meta_keys = get_post_custom_keys( $id );
	foreach ( $meta_keys as $meta_key ) {
		$values = get_post_custom_values( $meta_key, $id );
		foreach ( $values as $value ) {
			update_post_meta( $new, $meta_key, wp_slash( maybe_unserialize( $value ) ) );
		}
	}
	if ( is_plugin_active( 'elementor/elementor.php' ) && class_exists( '\Elementor\Core\Files\CSS\Post' ) ) {
		$elementor_css = call_user_func( array( '\Elementor\Core\Files\CSS\Post', 'create' ), $new );
		if ( $elementor_css && method_exists( $elementor_css, 'update' ) ) {
			$elementor_css->update();
		}
	}
	wp_redirect( "edit.php?post_type={$post->post_type}" );
	exit;
}

/**
 * Add duplicate to the posts page row actions in the admin
 *
 * @param array    $actions The action links array.
 * @param \WP_Post $post The post object.
 * @return array
 */
function duplicate_post_to_post_row_actions( $actions, $post ) {
	if ( empty( $post ) || ! current_user_can( get_post_type_object( $post->post_type )->cap->edit_posts ) || strstr( $post->post_type, 'acf-' ) ) {
		return $actions;
	}
	$id                   = $post->ID;
	$nonce                = wp_create_nonce( "kdweb-duplicate-{$id}" );
	$title                = __( 'Duplicate', 'kdweb' );
	$actions['duplicate'] = "<a href=\"admin.php?action=kdweb_duplicate_post&post={$id}&nonce={$nonce}\" rel=\"permalink\">{$title}</a>";
	return $actions;
}

/**
 * Enqueue assets for the edit page
 *
 * @return void
 */
function duplicate_post_admin_assets() {
	global $post;
	$screen = get_current_screen();
	if ( empty( $post ) || empty( $screen ) ) {
		return;
	}
	if ( method_exists( $screen, 'is_block_editor' ) && $screen->is_block_editor() ) {
		wp_enqueue_script( 'kdweb-duplicate-post', plugin_dir_url( __FILE__ ) . 'assets/editor-duplicate-post.js', array( 'wp-edit-post', 'wp-plugins', 'wp-i18n', 'wp-element' ), '1.0.0', true );
		wp_localize_script(
			'kdweb-duplicate-post',
			'kdweb_duplicate_post',
			array(
				'id'    => $post->ID,
				'nonce' => wp_create_nonce( "kdweb-duplicate-{$post->ID}" ),
				'title' => __( 'Duplicate', 'kdweb' ),
			)
		);
	} else {
		wp_enqueue_style( 'kdweb-duplicate-post', plugin_dir_url( __FILE__ ) . 'assets/edit-page.css', array(), '1.0.0' );
	}
}

/**
 * Add action link to the publish panel
 *
 * @return void
 */
function duplicate_post_to_edit_page() {
	global $post;
	$id    = $post->ID;
	$nonce = wp_create_nonce( "kdweb-duplicate-{$id}" );
	$title = __( 'Duplicate', 'kdweb' );
	?>
	<div class="misc-pub-section kdweb-duplicate-post">
		<span class="dashicons dashicons-clipboard"></span>
		<a href="admin.php?action=kdweb_duplicate_post&post=<?php echo esc_attr( $id ); ?>&nonce=<?php echo esc_attr( $nonce ); ?>" rel="permalink">
			<?php echo esc_html( $title ); ?>
		</a>
	</div>
	<?php
}

/**
 * Check for plugin updates.
 *
 * @param object $transient The transient object.
 * @return object The updated transient object.
 */
function check_plugin_update( $transient ) {
	$current_version = get_file_data( __FILE__, array( 'Version' => 'Version' ) )['Version'];
	$plugin_info = json_decode(
		wp_remote_retrieve_body(
			wp_remote_get( 'https://plugins.kddev.co.uk/kdweb-duplicate/update' )
		),
		true
	);
	if ( $plugin_info && ! is_wp_error( $plugin_info ) && version_compare( $plugin_info['new_version'], $current_version, '>' ) ) {
		$transient->response['kdweb-duplicate/kdweb-duplicate.php'] = (object) $plugin_info;
	}
	$transient->no_update['kdweb-duplicate/kdweb-duplicate.php'] = (object) array(
		'id'            => 'kdweb-duplicate/kdweb-duplicate.php',
		'slug'          => 'kdweb-duplicate',
		'plugin'        => 'kdweb-duplicate/kdweb-duplicate.php',
		'new_version'   => get_file_data( __FILE__, array( 'Version' => 'Version' ) )['Version'],
		'url'           => '',
		'package'       => '',
		'icons'         => array(),
		'banners'       => array(),
		'banners_rtl'   => array(),
		'tested'        => '',
		'requires_php'  => '',
		'compatibility' => new \stdClass(),
	);
	return $transient;
}
add_filter( 'pre_set_site_transient_update_plugins', 'KDWeb\Plugin\Duplicate\check_plugin_update' );

/**
 * Get plugin information from the plugin server.
 *
 * @param bool   $result The default value.
 * @param string $action The action to perform.
 * @param object $args The arguments for the action.
 * @return object|bool The plugin information or false.
 */
function plugin_info( $result, $action, $args ) {
	if ( 'plugin_information' === $action && 'kdweb-duplicate' === $args->slug ) {
		$plugin_info = json_decode(
			wp_remote_retrieve_body(
				wp_remote_get( 'https://plugins.kddev.co.uk/kdweb-duplicate/info' )
			),
			true
		);
		if ( ! $plugin_info || is_wp_error( $plugin_info ) ) {
			return $result;
		}
		return (object) $plugin_info;
	}
	return $result;
}
add_filter( 'plugins_api', 'KDWeb\Plugin\Duplicate\plugin_info', 10, 3 );
