<?php
/**
 * Plugin Name: Disable Comments
 * Plugin URI: https://plugins.kddev.co.uk/kdweb-disable-comments/
 * Description: Completely disable comments on your WordPress site, removing the feature completely.
 * Version: 1.0.0
 * Author: KD Web
 * Author URI: https://www.kddev.co.uk
 * Requires at least: 6.0.0
 * Requires PHP: 8.0.0
 *
 * @package KDWeb\Plugin\DisableComments
 */

namespace KDWeb\Plugin\DisableComments;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

add_filter( 'xmlrpc_methods', 'KDWeb\Plugin\DisableComments\disable_xmlprc_comments' );
add_filter( 'rest_endpoints', 'KDWeb\Plugin\DisableComments\disable_comments_filter_rest_endpoints' );
add_filter( 'rest_pre_insert_comment', 'KDWeb\Plugin\DisableComments\disable_comments_rest_api' );
$disable_comments_for_all_post_types = should_disable_all_post_type_comments();
if ( $disable_comments_for_all_post_types ) {
	add_filter( 'wp_headers', 'KDWeb\Plugin\DisableComments\disable_comments_filter_headers' );
	add_action( 'template_redirect', 'KDWeb\Plugin\DisableComments\disable_comments_in_feed_queries', 9 );
	add_action( 'template_redirect', 'KDWeb\Plugin\DisableComments\disable_comments_filter_admin_bar' );
	add_action( 'admin_init', 'KDWeb\Plugin\DisableComments\disable_comments_filter_admin_bar' );
}
add_action( 'widgets_init', 'KDWeb\Plugin\DisableComments\disable_comments_rc_widget' );
add_action( 'wp_loaded', 'KDWeb\Plugin\DisableComments\disable_comments_apply_filters' );
add_action( 'pre_get_comments', 'KDWeb\Plugin\DisableComments\filter_comments_queries' );
if ( is_admin() ) {
	add_action( 'enqueue_block_editor_assets', 'KDWeb\Plugin\DisableComments\disable_comments_blocks' );
}

/**
 * Apply additional filters after wp is loaded
 *
 * @return void
 */
function disable_comments_apply_filters() {
    $disable_comments_for_all_post_types = should_disable_all_post_type_comments();
    $post_types = apply_filters( 'kdweb_disable_comments_post_types', 'all' );
    if ( 'all' === $post_types ) {
        $post_types = get_post_types( array(), 'names' );
    }
    foreach ( $post_types as $post_type ) {
        if ( post_type_supports( $post_type, 'comments' ) ) {
            remove_post_type_support( $post_type, 'comments' );
            remove_post_type_support( $post_type, 'trackbacks' );
        }
    }
    add_filter( 'comments_array', 'KDWeb\Plugin\DisableComments\disable_comments_filter_existing_comments', 20, 2 );
    add_filter( 'comments_open', 'KDWeb\Plugin\DisableComments\disable_comments_filter_comment_status', 20, 2 );
    add_filter( 'pings_open', 'KDWeb\Plugin\DisableComments\disable_comments_filter_comment_status', 20, 2 );
    add_filter( 'get_comments_number', 'KDWeb\Plugin\DisableComments\disable_comments_filter_comments_count', 20, 2 );

    if ( is_admin() ) {
        if ( $disable_comments_for_all_post_types ) {
            add_action( 'admin_menu', 'KDWeb\Plugin\DisableComments\disable_comments_filter_admin_menu', PHP_INT_MAX );
            add_action( 'admin_print_styles-index.php', 'KDWeb\Plugin\DisableComments\disable_comments_admin_css' );
            add_action( 'admin_print_styles-profile.php', 'KDWeb\Plugin\DisableComments\disable_comments_admin_css' );
            add_action( 'wp_dashboard_setup', 'KDWeb\Plugin\DisableComments\disable_comments_filter_dashboard' );
            add_filter( 'pre_option_default_pingback_flag', '__return_zero' );
        }
    } else {
        add_action( 'template_redirect', 'KDWeb\Plugin\DisableComments\disable_comment_template_frontend' );
        if ( $disable_comments_for_all_post_types ) {
            add_filter( 'feed_links_show_comments_feed', '__return_false' );
        }
    }
}

/**
 * Disable latest comments block
 *
 * @return void
 */
function disable_comments_blocks() {
    global $post;
    if ( ! isset( $post->post_type ) || should_disable_comments_for_post_type( $post->post_type ) ) {
        wp_enqueue_script( 'kdweb-disable-comments-gutenberg',  plugin_dir_url( __FILE__ ) . 'assets/disable-comments.js', array(), false, true );
    }
}

/**
 * Remove comment method from xmlprc
 *
 * @param array $methods Available methods through the api.
 * @return array
 */
function disable_xmlprc_comments( $methods ) {
    unset($methods['wp.newComment']);
    return $methods;
}

/**
 * Remove pingback http header
 *
 * @param array $headers Default headers.
 * @return array
 */
function disable_comments_filter_headers( $headers ) {
    unset($headers['X-Pingback']);
    return $headers;
}

/**
 * Remove comments from the available rest api endpoints
 *
 * @param array $endpoints Available endpoints
 * @return array
 */
function disable_comments_filter_rest_endpoints( $endpoints ) {
    if ( isset( $endpoints['comments'] ) ) {
        unset( $endpoints['comments'] );
    }
    if ( isset( $endpoints['/wp/v2/comments'] ) ) {
        unset( $endpoints['/wp/v2/comments'] );
    }
    if ( isset( $endpoints['/wp/v2/comments/(?P<id>[\d]+)'] ) ) {
        unset( $endpoints['/wp/v2/comments/(?P<id>[\d]+)'] );
    }
    return $endpoints;
}

/**
 * Disable rest api comments
 *
 * @return void
 */
function disable_comments_rest_api() {
    return;
}

/**
 * Disable feed queries for comments
 *
 * @return void
 */
function disable_comments_in_feed_queries() {
    if (is_comment_feed()) {
        wp_die(
            __( 'Comments are closed.', 'kdweb' ),
            '',
            array( 'response' => 403 )
        );
    }
}

/**
 * Disable admin bar actions for comments
 *
 * @return void
 */
function disable_comments_filter_admin_bar() {
    if ( is_admin_bar_showing() ) {
        remove_action( 'admin_bar_menu', 'wp_admin_bar_comments_menu', 60 );
        if ( is_multisite() ) {
            add_action( 'admin_bar_menu', 'KDWeb\Plugin\DisableComments\disable_comments_remove_network_comment_links', 500 );
        }
    }
}

/**
 * Disable admin bar actions for comments on multisite setup
 *
 * @return void
 */
function disable_comments_remove_network_comment_links() {
	global $wp_admin_bar;
    foreach ( $wp_admin_bar->user->blogs ?? array() as $site ) {
        $wp_admin_bar->remove_menu( "blog-{$site->userblog_id}-c" );
    }
}

/**
 * Remove comment widget
 *
 * @return void
 */
function disable_comments_rc_widget() {
    unregister_widget( 'WP_Widget_Recent_Comments' );
    add_filter( 'show_recent_comments_widget_style', '__return_false' );
}

/**
 * Filter out comments when there is a comment query
 *
 * @param [type] $query
 * @return void
 */
function filter_comments_queries( $query ) {
	$query->query_vars['status'] = 'not-exists';
}

/**
 * Filter out comments for a post
 *
 * @param array $comments The comments to filter.
 * @param int $post_id The post ID.
 * @return array
 */
function disable_comments_filter_existing_comments( $comments, $post_id ) {
    $post_type = get_post_type( $post_id );
    return should_disable_comments_for_post_type( $post_type ) ? array() : $comments;
}

/**
 * Filter comment status for a post
 *
 * @param array $open Whether comments are open for the post.
 * @param int $post_id The post ID.
 * @return array
 */
function disable_comments_filter_comment_status( $open, $post_id ) {
    $post_type = get_post_type( $post_id );
    return should_disable_comments_for_post_type( $post_type ) ? false : $open;
}

/**
 * Filter comment count for a post
 *
 * @param array $count The comment count.
 * @param int $post_id The post ID.
 * @return array
 */
function disable_comments_filter_comments_count( $count, $post_id ) {
    $post_type = get_post_type( $post_id );
    return should_disable_comments_for_post_type( $post_type ) ? 0 : $count;
}

/**
 * Disable comments template frontend
 *
 * @return void
 */
function disable_comment_template_frontend() {
    if ( ! is_singular() || ! should_disable_comments_for_post_type( get_post_type() ) ) {
        return;
    }
    add_filter( 'comments_template', function() { return plugin_dir_path( __FILE__ ) . 'assets/empty-template.php'; }, 20 );
    wp_deregister_script( 'comment-reply' );
    remove_action( 'wp_head', 'feed_links_extra', 3 );
}

/**
 * Add snippet to admin to hide comments visually
 *
 * @return void
 */
function disable_comments_admin_css() {
    return '<style>#dashboard_right_now .comment-count,#dashboard_right_now .comment-mod-count,#latest-comments,#welcome-panel .welcome-comments,.user-comment-shortcuts-wrap{display:none!important;}</style>';
}

/**
 * Filter admin menu to remove comments
 *
 * @return void
 */
function disable_comments_filter_admin_menu() {
    global $pagenow;
    if ( 'comment.php' === $pagenow || 'edit-comments.php' === $pagenow || 'options-discussion.php' === $pagenow ) {
        wp_die(
            __( 'Comments are closed.', 'kdweb' ),
            '',
            array( 'response' => 403 )
        );
    }
    remove_menu_page( 'edit-comments.php' );
    remove_submenu_page( 'options-general.php', 'options-discussion.php' );
}

/**
 * Filter dashboard to remove comments box
 *
 * @return void
 */
function disable_comments_filter_dashboard() {
    remove_meta_box( 'dashboard_recent_comments', 'dashboard', 'normal' );
}

/**
 * Check whether comments should be disable for a specific post type
 *
 * @param string $post_type The post type.
 * @return boolean
 */
function should_disable_comments_for_post_type( $post_type = 'post' ) {
    $post_types = apply_filters( 'kdweb_disable_comments_post_types', 'all' );
    if ( 'all' === $post_types ) {
        return true;
    }
    if ( ! is_array( $post_types ) ) {
        return false;
    }
    return in_array( $post_type, $post_types, true );
}

/**
 * Check whether to disable comments everywhere
 *
 * @return boolean
 */
function should_disable_all_post_type_comments() {
    return 'all' === apply_filters( 'kdweb_disable_comments_post_types', 'all' );
}


/**
 * Check for plugin updates.
 *
 * @param object $transient The transient object.
 * @return object The updated transient object.
 */
function check_plugin_update( $transient ) {
	$current_version = get_file_data( __FILE__, array( 'Version' => 'Version' ) )['Version'];
	$plugin_info = json_decode(
		wp_remote_retrieve_body(
			wp_remote_get( 'https://plugins.kddev.co.uk/kdweb-disable-comments/update' )
		),
		true
	);
	if ( $plugin_info && ! is_wp_error( $plugin_info ) && version_compare( $plugin_info['new_version'], $current_version, '>' ) ) {
		$transient->response['kdweb-disable-comments/kdweb-disable-comments.php'] = (object) $plugin_info;
	}
	$transient->no_update['kdweb-disable-comments/kdweb-disable-comments.php'] = (object) array(
		'id'            => 'kdweb-disable-comments/kdweb-disable-comments.php',
		'slug'          => 'kdweb-disable-comments',
		'plugin'        => 'kdweb-disable-comments/kdweb-disable-comments.php',
		'new_version'   => get_file_data( __FILE__, array( 'Version' => 'Version' ) )['Version'],
		'url'           => '',
		'package'       => '',
		'icons'         => array(),
		'banners'       => array(),
		'banners_rtl'   => array(),
		'tested'        => '',
		'requires_php'  => '',
		'compatibility' => new \stdClass(),
	);
	return $transient;
}
add_filter( 'pre_set_site_transient_update_plugins', 'KDWeb\Plugin\DisableComments\check_plugin_update' );

/**
 * Get plugin information from the plugin server.
 *
 * @param bool   $result The default value.
 * @param string $action The action to perform.
 * @param object $args The arguments for the action.
 * @return object|bool The plugin information or false.
 */
function plugin_info( $result, $action, $args ) {
	if ( 'plugin_information' === $action && 'kdweb-disable-comments' === $args->slug ) {
		$plugin_info = json_decode(
			wp_remote_retrieve_body(
				wp_remote_get( 'https://plugins.kddev.co.uk/kdweb-disable-comments/info' )
			),
			true
		);
		if ( ! $plugin_info || is_wp_error( $plugin_info ) ) {
			return $result;
		}
		return (object) $plugin_info;
	}
	return $result;
}
add_filter( 'plugins_api', 'KDWeb\Plugin\DisableComments\plugin_info', 10, 3 );
