<?php
/**
 * Plugin Name: Admin
 * Plugin URI: https://plugins.kddev.co.uk/kdweb-admin/
 * Description: A collection of best practices, admin enhancements, white labelling and clutter removal for WordPress admin and html output.
 * Version: 1.0.0
 * Author: KD Web
 * Author URI: https://www.kddev.co.uk
 * Requires at least: 6.0.0
 * Requires PHP: 8.0.0
 *
 * @package KDWeb\Plugin\Admin
 */

namespace KDWeb\Plugin\Admin;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Load settings page
require_once __DIR__ . '/includes/settings-page.php';

/**
 * Plugin activation hook - apply default settings on first install.
 *
 * @return void
 */
function activate_plugin() {
	// Check if this is the first activation by looking for the enabled_features option
	if ( false === get_option( 'kdweb_admin_enabled_features' ) ) {
		// This is the first activation, apply defaults
		$defaults = get_default_settings();
		foreach ( $defaults as $option_name => $default_value ) {
			update_option( $option_name, $default_value );
		}
	}
}
register_activation_hook( __FILE__, 'KDWeb\Plugin\Admin\activate_plugin' );

// Load features
$enabled_features = get_option( 'kdweb_admin_enabled_features', array() );

// Undeletable Content (always loaded)
require_once __DIR__ . '/includes/undeletable-content.php';
add_filter( 'user_has_cap', 'KDWeb\Plugin\Admin\undeletable_posts_capabilities', 10, 3 );
add_filter( 'user_has_cap', 'KDWeb\Plugin\Admin\undeletable_terms_capabilities', 10, 3 );

// Hide Blocks (always loaded)
require_once __DIR__ . '/includes/hide-blocks.php';

// Admin Role
if ( in_array( 'create_admin_role', $enabled_features, true ) ) {
	require_once __DIR__ . '/includes/create-admin-role.php';
	add_action( 'admin_init', 'KDWeb\Plugin\Admin\register_admin_role' );
	add_filter( 'map_meta_cap', 'KDWeb\Plugin\Admin\admin_role_map_meta_cap', 10, 4 );
	add_filter( 'editable_roles', 'KDWeb\Plugin\Admin\admin_role_limit_editable_roles' );
	add_action( 'pre_user_query', 'KDWeb\Plugin\Admin\admin_role_hide_administrators_for_lower_level_users' );
	add_filter( 'views_users', 'KDWeb\Plugin\Admin\admin_role_exclude_administrator_filter_for_lower_users' );
	add_action( 'admin_menu', 'KDWeb\Plugin\Admin\admin_role_allow_polylang' );
	add_filter( 'redirection_role', 'KDWeb\Plugin\Admin\admin_role_allow_access' );
	add_filter( 'facetwp_admin_settings_capability', 'KDWeb\Plugin\Admin\admin_role_allow_access' );
	add_filter( 'relevanssi_options_capability', 'KDWeb\Plugin\Admin\admin_role_allow_access' );
}

// Disable Asset Versioning
if ( in_array( 'disable_asset_versioning', $enabled_features, true ) ) {
	require_once __DIR__ . '/includes/disable-asset-versioning.php';
	if ( get_option( 'kdweb_admin_disable_asset_versioning_only_unset', true ) ) {
		add_filter( 'script_loader_src', 'KDWeb\Plugin\Admin\disable_automatic_asset_version', 999 );
		add_filter( 'style_loader_src', 'KDWeb\Plugin\Admin\disable_automatic_asset_version', 999 );
	} else {
		add_filter( 'script_loader_src', 'KDWeb\Plugin\Admin\disable_asset_version', 999 );
		add_filter( 'style_loader_src', 'KDWeb\Plugin\Admin\disable_asset_version', 999 );
	}
}

// Disable Auto Updates
if ( in_array( 'disable_auto_updates', $enabled_features, true ) ) {
	require_once __DIR__ . '/includes/disable-auto-updates.php';
	if ( get_option( 'kdweb_admin_disable_auto_updates_core', true ) ) {
		add_filter( 'auto_update_core', '__return_false' );
		add_filter( 'site_status_tests', 'KDWeb\Plugin\Admin\dont_warn_about_auto_updates_on_site_health_check' );
	}
	if ( get_option( 'kdweb_admin_disable_auto_updates_minor', true ) ) {
		add_filter( 'allow_minor_auto_core_updates', '__return_false' );
		add_filter( 'site_status_tests', 'KDWeb\Plugin\Admin\dont_warn_about_auto_updates_on_site_health_check' );
	}
	if ( get_option( 'kdweb_admin_disable_auto_updates_plugin', true ) ) {
		add_filter( 'auto_update_plugin', '__return_false' );
		add_filter( 'plugins_auto_update_enabled', '__return_false' );
	}
	if ( get_option( 'kdweb_admin_disable_auto_updates_theme', true ) ) {
		add_filter( 'auto_update_theme', '__return_false' );
		add_filter( 'themes_auto_update_enabled', '__return_false' );
	}
	if ( get_option( 'kdweb_admin_disable_auto_updates_translation', false ) ) {
		add_filter( 'auto_update_translation', '__return_false' );
	}
}

// Disable Custom CSS
if ( in_array( 'disable_custom_css', $enabled_features, true ) ) {
	require_once __DIR__ . '/includes/disable-custom-css.php';
	add_action( 'customize_register', 'KDWeb\Plugin\Admin\disable_custom_css_in_customizer', 20 );
	remove_action( 'wp_head', 'wp_custom_css_cb', 11 );
	remove_action( 'wp_head', 'wp_custom_css_cb', 101 );
}

// Disable Customizer
if ( in_array( 'disable_customizer', $enabled_features, true ) ) {
	require_once __DIR__ . '/includes/disable-customizer.php';
	add_action( 'init', 'KDWeb\Plugin\Admin\disable_customizer_frontend', 10 );
	add_action( 'admin_init', 'KDWeb\Plugin\Admin\disable_customizer_admin', 10 );
}

// Disable Feeds
if ( in_array( 'disable_feeds', $enabled_features, true ) ) {
	require_once __DIR__ . '/includes/disable-feeds.php';
	add_action( 'do_feed', 'KDWeb\Plugin\Admin\disable_feeds_kill_when_feed', -1 );
	add_action( 'do_feed_rdf', 'KDWeb\Plugin\Admin\disable_feeds_kill_when_feed', -1 );
	add_action( 'do_feed_rss', 'KDWeb\Plugin\Admin\disable_feeds_kill_when_feed', -1 );
	add_action( 'do_feed_rss2', 'KDWeb\Plugin\Admin\disable_feeds_kill_when_feed', -1 );
	add_action( 'do_feed_atom', 'KDWeb\Plugin\Admin\disable_feeds_kill_when_feed', -1 );
	add_action( 'do_feed_rss2_comments', 'KDWeb\Plugin\Admin\disable_feeds_kill_when_feed', -1 );
	add_action( 'do_feed_atom_comments', 'KDWeb\Plugin\Admin\disable_feeds_kill_when_feed', -1 );
	add_action( 'init', 'KDWeb\Plugin\Admin\disable_feeds_remove_feed_rewrites' );
	remove_filter( 'wp_head', 'feed_links', 2 );
	remove_filter( 'wp_head', 'feed_links_extra', 3 );
}

// Disable JSON API
if ( in_array( 'disable_json_api', $enabled_features, true ) ) {
	require_once __DIR__ . '/includes/disable-json-api.php';
	if ( get_option( 'kdweb_admin_disable_json_api_for_unauthenticated_only', true ) ) {
		// Conditional logic handled in the feature file
		add_action( 'init', 'KDWeb\Plugin\Admin\init_disable_json_api' );
	} else {
		remove_action( 'xmlrpc_rsd_apis', 'rest_output_rsd' );
		remove_action( 'wp_head', 'rest_output_link_wp_head', 10 );
		remove_action( 'template_redirect', 'rest_output_link_header', 11 );
		add_filter( 'rest_authentication_errors', 'KDWeb\Plugin\Admin\disable_json_api_return_api_error', 20 );
		add_filter( 'site_status_tests', 'KDWeb\Plugin\Admin\disable_json_api_filter_out_site_health_check' );
	}
}

// Disable XML-RPC
if ( in_array( 'disable_xmlprc', $enabled_features, true ) ) {
	require_once __DIR__ . '/includes/disable-xmlprc.php';
	add_filter( 'xmlrpc_enabled', '__return_false' );
}

// Extend Sticky Posts
if ( in_array( 'extend_sticky_posts', $enabled_features, true ) ) {
	require_once __DIR__ . '/includes/extend-sticky-posts.php';
	add_filter( 'posts_orderby', 'KDWeb\Plugin\Admin\sticky_posts_orderby', 9, 2 );
	add_filter( 'post_class', 'KDWeb\Plugin\Admin\sticky_posts_add_class', 9, 3 );
}

// Load jQuery from CDN
if ( in_array( 'load_jquery_from_cdn', $enabled_features, true ) ) {
	require_once __DIR__ . '/includes/load-jquery-from-cdn.php';
	add_action( 'wp_enqueue_scripts', 'KDWeb\Plugin\Admin\load_jquery_from_cdn_reregister_jquery', 100 );
	add_filter( 'wp_resource_hints', 'KDWeb\Plugin\Admin\load_jquery_from_cdn_prefetch', 10, 2 );
}

// Setup Customized Login Page
if ( in_array( 'setup_customized_login_page', $enabled_features, true ) ) {
	require_once __DIR__ . '/includes/setup-customized-login-page.php';
	if ( get_option( 'kdweb_admin_setup_customized_login_page_logo', false ) ) {
		add_action( 'login_enqueue_scripts', 'KDWeb\Plugin\Admin\add_login_logo' );
	}
	if ( get_option( 'kdweb_admin_setup_customized_login_page_color', '' ) ) {
		add_action( 'login_enqueue_scripts', 'KDWeb\Plugin\Admin\add_login_color' );
	}
}

// Dashboard Info Widget
if ( in_array( 'setup_dashboard_info', $enabled_features, true ) ) {
	require_once __DIR__ . '/includes/setup-dashboard-info.php';
	add_action( 'wp_dashboard_setup', 'KDWeb\Plugin\Admin\register_kdweb_dashboard_info_widget' );
	add_action( 'admin_enqueue_scripts', 'KDWeb\Plugin\Admin\dashboard_info_styles' );
}

// Dashboard Launchpad Widget
if ( in_array( 'setup_dashboard_launchpad', $enabled_features, true ) ) {
	require_once __DIR__ . '/includes/setup-dashboard-launchpad.php';
	add_action( 'wp_dashboard_setup', 'KDWeb\Plugin\Admin\register_kdweb_dashboard_launchpad_widget' );
	add_action( 'admin_enqueue_scripts', 'KDWeb\Plugin\Admin\dashboard_launchpad_styles' );
}

// Logo Settings
if ( in_array( 'setup_logo_settings', $enabled_features, true ) ) {
	require_once __DIR__ . '/includes/setup-logo-settings.php';
	add_action( 'admin_init', 'KDWeb\Plugin\Admin\register_logo_settings' );
	add_action( 'admin_enqueue_scripts', 'KDWeb\Plugin\Admin\logo_settings_placement' );
	if ( get_option( 'kdweb_admin_setup_logo_settings_icon', false ) ) {
		add_action( 'admin_init', 'KDWeb\Plugin\Admin\logo_settings_save_icon' );
	}
}

// Simplify Admin
if ( in_array( 'simplify_admin', $enabled_features, true ) ) {
	require_once __DIR__ . '/includes/simplify-admin.php';
	if ( get_option( 'kdweb_admin_simplify_admin_admin_bar', true ) ) {
		add_action( 'admin_bar_menu', 'KDWeb\Plugin\Admin\simplify_admin_bar_nodes', 999 );
	}
	if ( get_option( 'kdweb_admin_simplify_admin_dashboard', true ) ) {
		add_action( 'wp_dashboard_setup', 'KDWeb\Plugin\Admin\simplify_admin_dashboard' );
	}
	if ( get_option( 'kdweb_admin_simplify_admin_taxonomy_checked_ontop', true ) ) {
		add_filter( 'wp_terms_checklist_args', 'KDWeb\Plugin\Admin\disable_taxonomy_checked_ontop' );
	}
	if ( get_option( 'kdweb_admin_simplify_admin_image_options', true ) ) {
		add_action( 'admin_init', 'KDWeb\Plugin\Admin\simplify_image_options' );
	}
	if ( get_option( 'kdweb_admin_simplify_admin_menus', true ) ) {
		add_action( 'admin_init', 'KDWeb\Plugin\Admin\admin_menu_menus_adjust_parent' );
		add_action( 'admin_menu', 'KDWeb\Plugin\Admin\admin_menu_add_menus' );
	}
	if ( get_option( 'kdweb_admin_simplify_admin_redirection', true ) ) {
		add_action( 'admin_init', 'KDWeb\Plugin\Admin\admin_menu_redirection_adjust_parent' );
		add_action( 'admin_menu', 'KDWeb\Plugin\Admin\admin_menu_redirection' );
	}
	if ( get_option( 'kdweb_admin_simplify_admin_facetwp', true ) ) {
		add_action( 'admin_init', 'KDWeb\Plugin\Admin\admin_menu_facetwp_adjust_parent' );
		add_action( 'admin_menu', 'KDWeb\Plugin\Admin\admin_menu_facetwp' );
	}
	if ( get_option( 'kdweb_admin_simplify_admin_relevanssi', true ) ) {
		add_action( 'admin_init', 'KDWeb\Plugin\Admin\admin_menu_relevanssi_adjust_parent' );
		add_action( 'admin_menu', 'KDWeb\Plugin\Admin\admin_menu_relevanssi' );
	}
	if ( get_option( 'kdweb_admin_simplify_admin_hide_themes', true ) ) {
		add_action( 'admin_menu', 'KDWeb\Plugin\Admin\admin_menu_remove_themes', 999 );
	}
	if ( get_option( 'kdweb_admin_simplify_admin_hide_tools', false ) ) {
		add_action( 'admin_menu', 'KDWeb\Plugin\Admin\admin_menu_remove_tools', 999 );
	}
	if ( get_option( 'kdweb_admin_simplify_admin_hide_settings', false ) ) {
		add_action( 'admin_menu', 'KDWeb\Plugin\Admin\admin_menu_remove_settings', 999 );
	}
	if ( get_option( 'kdweb_admin_simplify_admin_hide_updates', true ) ) {
		add_action( 'admin_menu', 'KDWeb\Plugin\Admin\admin_menu_remove_updates', 999 );
	}
	if ( get_option( 'kdweb_admin_simplify_admin_hide_acf', false ) ) {
		add_action( 'admin_menu', 'KDWeb\Plugin\Admin\admin_menu_remove_acf', 999 );
	}
	if ( get_option( 'kdweb_admin_simplify_admin_hide_yoast_premium_pages', true ) ) {
		add_action( 'admin_menu', 'KDWeb\Plugin\Admin\admin_menu_remove_yoast_premium_pages', 999 );
	}
	if ( get_option( 'kdweb_admin_simplify_admin_fix_unqueriable_post_types', true ) ) {
		add_action( 'admin_bar_menu', 'KDWeb\Plugin\Admin\fix_unqueriable_post_view_post_link_from_admin_bar', 90 );
		add_filter( 'get_sample_permalink_html', 'KDWeb\Plugin\Admin\fix_unqueriable_post_buttons', 10, 2 );
		add_filter( 'page_row_actions', 'KDWeb\Plugin\Admin\fix_unqueriable_post_row_actions', 10, 2 );
		add_filter( 'post_row_actions', 'KDWeb\Plugin\Admin\fix_unqueriable_post_row_actions', 10, 2 );
		add_filter( 'post_rewrite_rules', 'KDWeb\Plugin\Admin\fix_unqueriable_permalink_structures' );
		add_filter( 'is_post_type_viewable', 'KDWeb\Plugin\Admin\fix_unqueriable_post_type_viewable', 10, 2 );
		add_action( 'pre_get_posts', 'KDWeb\Plugin\Admin\fix_unqueriable_post_type_queries' );
	}
	if ( get_option( 'kdweb_admin_simplify_admin_gf_submit_to_button', true ) ) {
		add_filter( 'gform_submit_button', 'KDWeb\Plugin\Admin\gf_submit_to_button', 10, 2 );
	}
	add_filter( 'admin_footer_text', 'KDWeb\Plugin\Admin\overwrite_admin_footer_thankyou', 99 );
	add_filter( 'login_headerurl', 'KDWeb\Plugin\Admin\login_logo_to_home' );
	add_filter( 'login_privacy_policy_url', 'KDWeb\Plugin\Admin\fix_login_privacy_policy_link', 99 );
}

// Simplify HTML Output
if ( in_array( 'simplify_html_output', $enabled_features, true ) ) {
	require_once __DIR__ . '/includes/simplify-html-output.php';
	$simple_mode = get_option( 'kdweb_admin_simplify_html_output_simple_mode', false );

	if ( $simple_mode || get_option( 'kdweb_admin_simplify_html_output_archive_title', true ) ) {
		add_filter( 'get_the_archive_title', 'KDWeb\Plugin\Admin\simpler_archive_title' );
	}
	if ( $simple_mode || get_option( 'kdweb_admin_simplify_html_output_archive_link', true ) ) {
		add_filter( 'get_archives_link', 'KDWeb\Plugin\Admin\simpler_get_archives_link', 10, 2 );
	}
	if ( $simple_mode || get_option( 'kdweb_admin_simplify_html_output_body_class', true ) ) {
		add_filter( 'body_class', 'KDWeb\Plugin\Admin\simpler_html_body_class', 999 );
	}
	if ( $simple_mode || get_option( 'kdweb_admin_simplify_html_output_category_class', true ) ) {
		add_filter( 'category_css_class', 'KDWeb\Plugin\Admin\simpler_html_category_css_class', 999, 4 );
	}
	if ( $simple_mode || get_option( 'kdweb_admin_simplify_html_output_comment_class', true ) ) {
		add_filter( 'comment_class', 'KDWeb\Plugin\Admin\simpler_html_comment_class', 999 );
	}
	if ( $simple_mode || get_option( 'kdweb_admin_simplify_html_output_image_attributes', true ) ) {
		add_filter( 'wp_get_attachment_image_attributes', 'KDWeb\Plugin\Admin\simpler_image_attr', 999, 3 );
	}
	if ( $simple_mode || get_option( 'kdweb_admin_simplify_html_output_menu_classes', true ) ) {
		add_filter( 'nav_menu_css_class', 'KDWeb\Plugin\Admin\simpler_html_menu_classes', 999 );
		add_filter( 'nav_menu_item_id', 'KDWeb\Plugin\Admin\simpler_html_menu_classes', 999 );
	}
	if ( $simple_mode || get_option( 'kdweb_admin_simplify_html_output_post_class', true ) ) {
		add_filter( 'post_class', 'KDWeb\Plugin\Admin\simpler_html_post_class', 999, 3 );
	}
	if ( $simple_mode || get_option( 'kdweb_admin_simplify_html_output_comment_form', true ) ) {
		add_filter( 'comment_form_defaults', 'KDWeb\Plugin\Admin\simpler_comment_args' );
	}
	if ( $simple_mode || get_option( 'kdweb_admin_simplify_html_output_comment_form_website', true ) ) {
		add_filter( 'comment_form_default_fields', 'KDWeb\Plugin\Admin\simpler_comment_form_remove_website_link' );
	}
	if ( $simple_mode || get_option( 'kdweb_admin_simplify_html_output_head_charset', true ) ) {
		remove_action( 'wp_head', 'wp_generator' );
		add_action( 'wp_head', 'KDWeb\Plugin\Admin\add_charset_to_wphead', 0 );
	}
	if ( $simple_mode || get_option( 'kdweb_admin_simplify_html_output_head_xua_compatible', true ) ) {
		add_action( 'wp_head', 'KDWeb\Plugin\Admin\add_edge_xua_to_wphead', 1 );
	}
	if ( $simple_mode || get_option( 'kdweb_admin_simplify_html_output_head_viewport', true ) ) {
		add_action( 'wp_head', 'KDWeb\Plugin\Admin\add_viewport_to_wphead', 1 );
	}
	if ( get_option( 'kdweb_admin_simplify_html_output_head_revisit_after', false ) ) {
		add_action( 'wp_head', 'KDWeb\Plugin\Admin\add_revisit_after_to_wphead', 1 );
	}
	if ( get_option( 'kdweb_admin_simplify_html_output_head_distribution', false ) ) {
		add_action( 'wp_head', 'KDWeb\Plugin\Admin\add_distribution_to_wphead', 1 );
	}
	if ( $simple_mode || get_option( 'kdweb_admin_simplify_html_output_head_wp_version', true ) ) {
		remove_action( 'wp_head', 'wp_generator' );
	}
	if ( $simple_mode || get_option( 'kdweb_admin_simplify_html_output_head_wlwmanifest_link', true ) ) {
		remove_action( 'wp_head', 'wlwmanifest_link' );
	}
	if ( $simple_mode || get_option( 'kdweb_admin_simplify_html_output_head_rsd_link', true ) ) {
		remove_action( 'wp_head', 'rsd_link' );
	}
	if ( $simple_mode || get_option( 'kdweb_admin_simplify_html_output_head_shortlink', true ) ) {
		remove_action( 'wp_head', 'wp_shortlink_wp_head' );
	}
	if ( $simple_mode || get_option( 'kdweb_admin_simplify_html_output_head_adjacent_posts_link', true ) ) {
		remove_action( 'wp_head', 'adjacent_posts_rel_link_wp_head' );
	}
	if ( $simple_mode || get_option( 'kdweb_admin_simplify_html_output_head_emoji', true ) ) {
		remove_action( 'wp_head', 'print_emoji_detection_script', 7 );
		remove_action( 'wp_print_styles', 'print_emoji_styles' );
	}
	if ( $simple_mode || get_option( 'kdweb_admin_simplify_html_output_head_oembed', true ) ) {
		remove_action( 'wp_head', 'wp_oembed_add_discovery_links' );
	}
	if ( $simple_mode || get_option( 'kdweb_admin_simplify_html_output_head_post_relational_links', true ) ) {
		remove_action( 'wp_head', 'start_post_rel_link' );
		remove_action( 'wp_head', 'index_rel_link' );
		remove_action( 'wp_head', 'parent_post_rel_link' );
	}
	if ( $simple_mode || get_option( 'kdweb_admin_simplify_html_output_head_core_block_styles', true ) ) {
		add_action( 'wp_enqueue_scripts', 'KDWeb\Plugin\Admin\remove_core_blocks_styles', 100 );
		add_filter( 'styles_inline_size_limit', 'KDWeb\Plugin\Admin\remove_core_blocks_css' );
		add_action( 'wp_footer', 'KDWeb\Plugin\Admin\remove_core_blocks_css_from_wp_footer' );
	}
	if ( $simple_mode || get_option( 'kdweb_admin_simplify_html_output_remove_menu_hash', true ) ) {
		add_filter( 'wp_nav_menu_args', 'KDWeb\Plugin\Admin\simpler_html_menu_args' );
		add_filter( 'wp_nav_menu_items', 'KDWeb\Plugin\Admin\remove_hash_anchors', 99, 2 );
	}
	if ( $simple_mode || get_option( 'kdweb_admin_simplify_html_output_html5_tags', true ) ) {
		add_filter( 'style_loader_tag', 'KDWeb\Plugin\Admin\html5_link_tag', 10 );
		add_filter( 'script_loader_tag', 'KDWeb\Plugin\Admin\html5_script_tag', 10 );
		add_filter( 'kdweb_theme_html', 'KDWeb\Plugin\Admin\self_closing_tags' );
	}
	if ( $simple_mode || get_option( 'kdweb_admin_simplify_html_output_normalize_classnames', true ) ) {
		add_filter( 'kdweb_theme_html', 'KDWeb\Plugin\Admin\normalize_classnames' );
	}
}

/**
 * Check for plugin updates.
 *
 * @param object $transient The transient object.
 * @return object The updated transient object.
 */
function check_plugin_update( $transient ) {
	$current_version = get_file_data( __FILE__, array( 'Version' => 'Version' ) )['Version'];
	$plugin_info = json_decode(
		wp_remote_retrieve_body(
			wp_remote_get( 'https://plugins.kddev.co.uk/kdweb-admin/update' )
		),
		true
	);
	if ( $plugin_info && ! is_wp_error( $plugin_info ) && version_compare( $plugin_info['new_version'], $current_version, '>' ) ) {
		$transient->response['kdweb-admin/kdweb-admin.php'] = (object) $plugin_info;
	}
	$transient->no_update['kdweb-admin/kdweb-admin.php'] = (object) array(
		'id'            => 'kdweb-admin/kdweb-admin.php',
		'slug'          => 'kdweb-admin',
		'plugin'        => 'kdweb-admin/kdweb-admin.php',
		'new_version'   => get_file_data( __FILE__, array( 'Version' => 'Version' ) )['Version'],
		'url'           => '',
		'package'       => '',
		'icons'         => array(),
		'banners'       => array(),
		'banners_rtl'   => array(),
		'tested'        => '',
		'requires_php'  => '',
		'compatibility' => new \stdClass(),
	);
	return $transient;
}
add_filter( 'pre_set_site_transient_update_plugins', 'KDWeb\Plugin\Admin\check_plugin_update' );

/**
 * Get plugin information from the plugin server.
 *
 * @param bool   $result The default value.
 * @param string $action The action to perform.
 * @param object $args The arguments for the action.
 * @return object|bool The plugin information or false.
 */
function plugin_info( $result, $action, $args ) {
	if ( 'plugin_information' === $action && 'kdweb-admin' === $args->slug ) {
		$plugin_info = json_decode(
			wp_remote_retrieve_body(
				wp_remote_get( 'https://plugins.kddev.co.uk/kdweb-admin/info' )
			),
			true
		);
		if ( ! $plugin_info || is_wp_error( $plugin_info ) ) {
			return $result;
		}
		return (object) $plugin_info;
	}
	return $result;
}
add_filter( 'plugins_api', 'KDWeb\Plugin\Admin\plugin_info', 10, 3 );
