<?php
/**
 * Undeletable Posts and Terms
 *
 * @package KDWeb\Plugin\Admin
 */

namespace KDWeb\Plugin\Admin;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Get key pages that should be undeletable
 *
 * @return array Array of post IDs
 */
function get_key_pages() {
	$key_pages = array();

	// Add homepage
	$page_on_front = get_option( 'page_on_front' );
	if ( $page_on_front ) {
		$key_pages[] = (int) $page_on_front;
	}

	// Add blog page
	$page_for_posts = get_option( 'page_for_posts' );
	if ( $page_for_posts ) {
		$key_pages[] = (int) $page_for_posts;
	}

	// Add WooCommerce pages if WooCommerce is active
	if ( class_exists( 'WooCommerce' ) ) {
		$woo_pages = array(
			'woocommerce_shop_page_id',
			'woocommerce_cart_page_id',
			'woocommerce_checkout_page_id',
			'woocommerce_myaccount_page_id',
			'woocommerce_terms_page_id',
		);

		foreach ( $woo_pages as $woo_page ) {
			$page_id = get_option( $woo_page );
			if ( $page_id ) {
				$key_pages[] = (int) $page_id;
			}
		}
	}

	return array_unique( array_filter( $key_pages ) );
}

/**
 * Make some posts undeletable in the admin
 *
 * @param array $posts An array of post IDs to be made undeletable.
 * @return void
 */
function make_posts_undeletable( $posts = array() ) {
	global $_kdweb_undeletable_posts;
	$_kdweb_undeletable_posts = $posts;
}

/**
 * Filter capabilities to remove user capability to delete some posts
 *
 * @param array $allcaps Array of key/value pairs where keys represent a capability name and boolean values represent whether the user has that capability.
 * @param array $caps    Required primitive capabilities for the requested capability.
 * @param array $args    Arguments that accompany the requested capability check.
 * @return array
 */
function undeletable_posts_capabilities( $allcaps, $caps, $args ) {
	global $_kdweb_undeletable_posts;
	if ( 'delete_post' !== $args[0] ) {
		return $allcaps;
	}

	// Get undeletable posts from settings
	$settings_posts = get_option( 'kdweb_admin_undeletable_posts', array() );

	// Support old comma-separated format
	if ( ! is_array( $settings_posts ) ) {
		$settings_posts = ! empty( $settings_posts ) ? array_map( 'intval', array_map( 'trim', explode( ',', $settings_posts ) ) ) : array();
	} else {
		$settings_posts = array_map( 'intval', $settings_posts );
	}

	// Add key pages if enabled
	if ( get_option( 'kdweb_admin_undeletable_key_pages', false ) ) {
		$key_pages = get_key_pages();
		$settings_posts = array_merge( $settings_posts, $key_pages );
	}

	// Merge with programmatically set posts and apply filter
	$undeletable_posts = apply_filters( 'kdweb_undeletable_posts', array_merge( $_kdweb_undeletable_posts ?? array(), $settings_posts ) );

	if ( in_array( $args[2], $undeletable_posts, true ) ) {
		$allcaps[ $args[0] ] = false;
		foreach ( $caps as $cap ) {
			$allcaps[ $cap ] = false;
		}
	}
	return $allcaps;
}

/**
 * Make some terms undeletable in the admin
 *
 * @param array $terms An array of terms IDs to be made undeletable.
 * @return void
 */
function make_terms_undeletable( $terms = array() ) {
	global $_kdweb_undeletable_terms;
	$_kdweb_undeletable_terms = $terms;
}

/**
 * Filter capabilities to remove user capability to delete some terms
 *
 * @param array $allcaps Array of key/value pairs where keys represent a capability name and boolean values represent whether the user has that capability.
 * @param array $caps    Required primitive capabilities for the requested capability.
 * @param array $args    Arguments that accompany the requested capability check.
 * @return array
 */
function undeletable_terms_capabilities( $allcaps, $caps, $args ) {
	global $_kdweb_undeletable_terms;
	if ( 'delete_term' !== $args[0] ) {
		return $allcaps;
	}

	// Get undeletable terms from settings
	$settings_terms = get_option( 'kdweb_admin_undeletable_terms', array() );

	// Support old comma-separated format
	if ( ! is_array( $settings_terms ) ) {
		$settings_terms = ! empty( $settings_terms ) ? array_map( 'intval', array_map( 'trim', explode( ',', $settings_terms ) ) ) : array();
	} else {
		$settings_terms = array_map( 'intval', $settings_terms );
	}

	// Merge with programmatically set terms and apply filter
	$undeletable_terms = apply_filters( 'kdweb_undeletable_terms', array_merge( $_kdweb_undeletable_terms ?? array(), $settings_terms ) );

	if ( in_array( $args[2], $undeletable_terms, true ) ) {
		$allcaps[ $args[0] ] = false;
		foreach ( $caps as $cap ) {
			$allcaps[ $cap ] = false;
		}
	}
	return $allcaps;
}
