<?php
/**
 * Simplify HTML output created by WordPress
 *
 * @package KDWeb\Plugin\Admin
 */

namespace KDWeb\Plugin\Admin;

if ( ! defined( 'ABSPATH' ) ) { exit; }

/**
 * Replace colon style archive headings with more simple ones
 *
 * @param string $title The title by default.
 * @return string
 */
function simpler_archive_title( $title ) {
	global $wp_query;
	if ( is_search() ) {
		return apply_filters(
            'kdweb_archive_title_search',
            wp_sprintf(
                /* translators: Search term */
                __( 'Search results for <strong>%s</strong>' ),
                get_query_var( 's' )
            ),
            get_query_var( 's' )
        );
	}
	if ( is_author() ) {
		return get_the_author();
	}
	if ( is_tax() || is_category() || is_tag() ) {
		return $wp_query->get_queried_object()->name;
	}
	if ( is_post_type_archive() ) {
		return $wp_query->get_queried_object()->label;
	}
	if ( is_date() ) {
		$title = explode( ': ', $title )[1];
		if ( isset( $wp_query->query['day'] ) ) {
			$time  = strtotime(
				$wp_query->query['year'] . '-' .
					$wp_query->query['monthnum'] . '-' .
					$wp_query->query['day']
			);
			$title = date_i18n( get_option( 'date_format' ), $time );
		}
	}
	if ( is_home() ) {
		return get_post_type_object( 'post' )->label;
	}
	return $title;
}

/**
 * Simplify archive link
 *
 * @param string $link_html The full HTML.
 * @param string $url The URL.
 * @return string
 */
function simpler_get_archives_link( $link_html, $url ) {
	if ( strstr( $link_html, '<li>' ) ) {
		global $wp_query;
		if ( $wp_query->is_date() ) {
			$year_month = '/' . $wp_query->query['year'];
			if ( isset( $wp_query->query['monthnum'] ) ) {
				$year_month .= '/' . $wp_query->query['monthnum'];
			}
			if ( strstr( $url, $year_month ) ) {
				$link_html = str_replace( '<li>', '<li class="is-active is-current">', $link_html );
			}
		}
	}
	return $link_html;
}

/**
 * Simplify body class
 *
 * @param array $classes Array of CSS classes.
 * @return array
 */
function simpler_html_body_class( $classes ) {
	global $_kdweb_simplify_simple_mode;
	foreach ( $classes as &$classname ) {
		$classname = preg_replace(
			array(
				'/^(archive|paged|page|single|post\-type\-archive|page\-template|no\-customize\-support|postid\-[^.]++|attachmentid\-[^.]+|parent\-pageid\-[^.]+|author\-[^.]+|category\-[0-9]+|tag\-[0-9]+|term\-[0-9]|[^.]+\-paged\-[0-9]+|[^.]+\-id\-[0-9]+|[^.]+\-template[^.]+\-php|wp-embed-responsive)$/i',
				'/^(rtl|author|home|blog|date|search)$/i',
				'/^(?:[^.]+)\-(parent|child)$/i',
				'/^paged-([0-9]+)$/i',
				'/^(?:post\-type\-archive|single)\-([^\-]+)$/i',
				'/^tax\-([^.]+)$/i',
				'/^(category|tag)$/i',
				'/^(?:category|tag|term)\-([^.]+)$/i',
				'/^search\-(no\-)?results$/i',
				'/^(?:[^\-]+)\-template(?:\-template)?\-([^\-]+)$/i',
				'/^single\-format\-([^.]+)$/i',
				'/^error404$/i',
				'/^logged-in$/i',
				'/^admin-bar$/i',
			),
			empty( $_kdweb_simplify_simple_mode ) ? array(
				'',
				'is-$1',
				'is-$1',
				'is-page-$1',
				'is-type-$1',
				'is-taxonomy-$1',
				'is-taxonomy-$1',
				'is-term-$1',
				'has-$1results',
				'template-$1',
				'is-format-$1',
				'',
				'has-user-logged-in',
				'has-admin-bar',
			) : array(
				'',
				'',
				'',
				'',
				'',
				'',
				'',
				'',
				'',
				'template-$1',
				'',
				'',
				'has-user-logged-in',
				'has-admin-bar',
			),
			$classname
		);
	}
	$classes = array_merge(
		array(
			is_404() ? 'type-error' : ( is_singular() ? 'type-single' : 'type-archive' ),
		),
		$classes
	);
	return normalize_classnames( $classes );
}

/**
 * Simplify category css class
 *
 * @param string|array $classes The classnames.
 * @param WP_Term      $category The category term.
 * @param int          $depth The depth.
 * @param array        $args The arguments.
 * @return array
 */
function simpler_html_category_css_class( $classes, $category, $depth, $args ) { // phpcs:ignore
	foreach ( $classes as &$classname ) {
		$classname = preg_replace(
			array( '/^cat\-item(\-[0-9]+)?$/i' ),
			array( '' ),
			$classname
		);
	}
	$classes = array_merge(
		$classes,
		array(
			'term',
			$category->slug,
			'taxonomy-' . $category->taxonomy,
		)
	);
	if ( isset( $args['active_class'] ) && $args['active_class'] ) {
		global $wp_query;
		if (
			$wp_query->get_queried_object() &&
			'WP_Term' === get_class( $wp_query->get_queried_object() ) &&
			$wp_query->get_queried_object()->term_id === $category->term_id
		) {
			$classes[] = 'is-active';
			$classes[] = 'is-current';
		}
	}
	return normalize_classnames( $classes );
}

/**
 * Simplify comment class
 *
 * @param array $classes Array of classnames.
 * @return array
 */
function simpler_html_comment_class( $classes ) {
	foreach ( $classes as &$classname ) {
		$classname = preg_replace(
			array(
				'/^byuser$/i',
				'/^bypostauthor$/i',
				'/^comment\-author\-([^.]+)$/i',
				'/^(even|odd|alt|thread\-even|thread\-odd|thread\-alt)$/',
				'/^depth-([0-9]+)$/i',
				'/^parent$/i',
			),
			array(
				'comment',
				'is-by-user',
				'is-by-author',
				'is-by-$1',
				'',
				'is-level-$1',
				'has-children',
			),
			$classname
		);
	}
	return normalize_classnames( $classes );
}

/**
 * Filter the attachment image class.
 *
 * @param string[]     $attr       Array of attribute values for the image markup, keyed by attribute name.
 * @param WP_Post      $attachment Image attachment post.
 * @param string|array $size     Requested size. Image size or array of width and height values (in that order). Default 'thumbnail'.
 * @return string[]
 */
function simpler_image_attr( $attr, $attachment, $size ) {
	$size_class = is_array( $size ) ? join( 'x', $size ) : $size;
	$classes    = array( 'img' );
	foreach ( is_array( $attr['class'] ) ? $attr['class'] : explode( ' ', $attr['class'] ) as $classname ) {
		$classname = preg_replace( '/^wp-post-image$/i', '', $classname );
		$classname = preg_replace( '/^size-post-thumbnail$/i', '', $classname );
		$classname = preg_replace( '/^attachment-([^.]+)$/i', '', $classname );
		$classname = preg_replace( '/^size-([^.]+)$/i', 'is-size-$1', $classname );
		$classes[] = $classname;
	}
	$dimensions = get_attachment_dimensions( $attachment->ID );
	if ( $dimensions ) {
		$classes[] = $dimensions['landscape'] ? 'is-landscape' : ( $dimensions['portrait'] ? 'is-portrait' : 'is-square' );
	}
	$attr['class'] = implode( ' ', normalize_classnames( $classes ) );
	return $attr;
}

/**
 * Get the width and height of an image from the media library
 *
 * @param int $id The attachment ID
 * @return null|array
 */
function get_attachment_dimensions( $id ) {
	$details = wp_get_attachment_image_src( $id );
	if ( empty( $details ) || ! array( $details ) || ! isset( $details[1] ) || ! isset( $details[2] ) ) {
		return null;
	}
	return array(
		'width'     => $details[1],
		'height'    => $details[2],
		'ratio'     => $details[1] / $details[2],
		'landscape' => $details[1] > $details[2],
		'portrait'  => $details[1] < $details[2],
		'square'    => $details[1] === $details[2],
	);
}

/**
 * Remove unnecessary WordPress classes from menu items.
 *
 * @param array $classes The array of CSS classes.
 * @return array
 */
function simpler_html_menu_classes( $classes ) {
	if ( ! is_array( $classes ) ) {
		$classes = explode( ' ', $classes );
	}
	foreach ( $classes as &$classname ) {
		$classname = preg_replace( '/^current(?:\-|\_)[^\-]+(?:\-|\_)item$/i', 'is-current is-active', $classname );
		$classname = preg_replace( '/^current(?:\-|\_)[^\-]+(?:\-|\_)parent$/i', 'is-parent has-current is-active', $classname );
		$classname = preg_replace( '/^current(?:\-|\_)[^\-]+(?:\-|\_)ancestor$/i', 'is-ancestor has-current is-active', $classname );
		$classname = preg_replace( '/^current\[^\-]+(?:\-|\_)item$/i', '', $classname );
		$classname = preg_replace( '/^current\[^\-]+(?:\-|\_)ancestor$/i', '', $classname );
		$classname = preg_replace( '/^menu(?:\-|\_)item(?:\-|\_)has(?:\-|\_)children$/i', 'has-children', $classname );
		$classname = preg_replace( '/^menu(?:\-|\_)item(?:\-|\_)type(?:\-|\_)taxonomy$/i', 'is-taxonomy', $classname );
		$classname = preg_replace( '/^menu(?:\-|\_)item(?:\-|\_)object(?:\-|\_)([^\-]+)$/i', 'is-$1', $classname );
		$classname = preg_replace( '/^menu(?:\-|\_)item$/i', 'menu-item', $classname );
		$classname = preg_replace( '/^menu(?:\-|\_)item(\-type\-[^\-]+)?(\-[0-9]+)?$/i', '', $classname );
		$classname = preg_replace( '/^page(?:\-|\_)item(\-[0-9]+)?$/i', '', $classname );
	}
	return normalize_classnames( $classes );
}

/**
 * Simplify post class
 *
 * @param string|array $classes The classes.
 * @param string|array $class The class as a string.
 * @param int          $id The post ID.
 * @return array
 */
function simpler_html_post_class( $classes, $class, $id ) { // phpcs:ignore
	$post_type  = preg_quote( get_post_type( $id ), null );
	$taxonomies = array_merge(
		array(
			'category',
			'tag',
		),
		get_taxonomies(
			array(
				'public'   => true,
				'_builtin' => false,
			)
		)
	);
	$taxonomies = join( '|', array_map( 'preg_quote', $taxonomies ) );
	foreach ( $classes as &$classname ) {
		$classname = preg_replace(
			array(
				'/^(hentry|post\-[0-9]+|status\-[^.]+)$/i',
				"/^(?:type-)?{$post_type}$/",
				'/^(format)\-([^.]+)$/i',
				"/^({$taxonomies})\-([^.]+)$/i",
			),
			array(
				'',
				"is-type-{$post_type}",
				'in-$1-$2',
				'in-$1-$2',
			),
			$classname
		);
	}
	return normalize_classnames( $classes );
}

/**
 * Simplify comments
 *
 * @param array $args The arguments.
 * @return array
 */
function simpler_comment_args( $args ) {
	if ( null === $args['walker'] ) {
		$args['walker'] = new \KDWeb\Plugin\Admin\Comment_Walker();
	}
	return $args;
}

/**
 * Remove the website link field from the comment form
 *
 * @param array $fields Form fields.
 * @return array
 */
function simpler_comment_form_remove_website_link( $fields ) {
	if ( isset( $fields['url'] ) ) {
		unset( $fields['url'] );
	}
	return $fields;
}

/**
 * Add information to <head> output about charset
 *
 * @return void
 */
function add_charset_to_wphead() {
	$charset      = get_bloginfo( 'charset' );
	$content_type = get_bloginfo( 'html_type' ) . "; charset={$charset}";
	?>
	<meta charset="<?php echo esc_attr( $charset ); ?>">
	<meta http-equiv="Content-Type" content="<?php echo esc_attr( $content_type ); ?>">
	<?php
}

/**
 * Add information to <head> output about x-ua-compatibility
 *
 * @return void
 */
function add_edge_xua_to_wphead() {
	?>
	<meta http-equiv="X-UA-Compatible" content="IE=edge">
	<?php
}

/**
 * Add information to <head> output about viewport settings
 *
 * @return void
 */
function add_viewport_to_wphead() {
	?>
	<meta name="MobileOptimized" content="320">
	<meta name="HandheldFriendly" content="true">
	<meta name="viewport" content="width=device-width, initial-scale=1.0">
	<?php
}

/**
 * Add information to <head> output about robots rule to revisit site
 *
 * @return void
 */
function add_revisit_after_to_wphead() {
	?>
	<meta name="revisit-after" content="30 days">
	<?php
}

/**
 * Add information to <head> output about content distribution type
 *
 * @return void
 */
function add_distribution_to_wphead() {
	?>
	<meta name="distribution" content="web">
	<?php
}

/**
 * Remove core blocks styles
 *
 * @return void
 */
function remove_core_blocks_styles() {
	remove_action( 'wp_enqueue_scripts', 'wp_enqueue_global_styles' );
	remove_action( 'wp_body_open', 'wp_global_styles_render_svg_filters' );
	remove_action( 'wp_enqueue_scripts', 'wp_enqueue_global_styles_custom_css' );
}

/**
 * Remove core blocks css output
 *
 * @return void
 */
function remove_core_blocks_css() {
	wp_dequeue_style( 'wp-block-library' );
	wp_dequeue_style( 'classic-theme-styles' );
}

/**
 * Remove core blocks css out from wp footer
 *
 * @return void
 */
function remove_core_blocks_css_from_wp_footer() {
	wp_dequeue_style( 'core-block-supports' );
}

/**
 * Simplify menus
 *
 * @param string|array $args The settings.
 * @return string|array
 */
function simpler_html_menu_args( $args = '' ) {
	if (
		'div' === $args['container'] &&
		empty( $args['container_id'] ) &&
		empty( $args['container_class'] )
	) {
		$args['container'] = false;
	}
	if ( 'wp_page_menu' === $args['fallback_cb'] ) {
		$args['fallback_cb'] = null;
	}
	if ( '<ul id="%1$s" class="%2$s">%3$s</ul>' === $args['items_wrap'] ) {
		$args['items_wrap'] = '<ul class="%2$s">%3$s</ul>';
	}
	if ( ! empty( $args['menu'] ) ) {
		$args['menu_class'] .= ' is-name-' . sanitize_title_with_dashes( $args['menu'] );
	}
	return $args;
}

/**
 * Remove anchor element for non-link menu items and replace them with spans
 *
 * @param string $output The menu item start.
 * @param object $item The menu item object.
 * @return string
 */
function remove_hash_anchors( $output, $item ) {
	if ( '#' !== $item->url ) {
		return $output;
	}
	return str_replace(
		array(
			'<a ',
			'</a>',
			' href="#"',
			" href='#'",
		),
		array(
			'<span ',
			'</span>',
			'',
			'',
		),
		$output
	);
}

/**
 * Remove unnecessary html4 script syntax
 *
 * @param string $text The output HTML for the scrip tag.
 * @return string
 */
function html5_script_tag( $text ) {
	if ( is_admin() ) {
		return $text;
	}
	$text = str_replace( "type='text/javascript' ", '', $text );
	return str_replace( "'", '"', $text );
}

/**
 * Remove unnecessary html4 link syntax
 *
 * @param string $text The output HTML for the link tag.
 * @return string
 */
function html5_link_tag( $text ) {
	if ( is_admin() ) {
		return $text;
	}
	$text = str_replace( "type='text/css' ", '', $text );
	return str_replace( "'", '"', $text );
}

/**
 * Remove slash from the end of self closing tags
 *
 * @param string $text The output text.
 * @return string
 */
function self_closing_tags( $text ) {
	return preg_replace( '/<([^>]+)\s\/>/i', '<$1>', $text );
}

/**
 * Normalize classnames arrays
 *
 * @param array $classnames The initial classnames array.
 * @return array
 */
function normalize_classnames( $classnames ) {
	$groups     = array(
		'def'   => array(),
		'is'    => array(),
		'in'    => array(),
		'on'    => array(),
		'by'    => array(),
		'with'  => array(),
		'has'   => array(),
		'other' => array(),
	);
	$classnames = array_unique( array_filter( explode( ' ', implode( ' ', $classnames ) ) ) );
	foreach ( $classnames as $key => $class ) {
		$where = 'other';
		if ( 'is-' === substr( $class, 0, 3 ) ) {
			$where = 'is';
		}
		if ( 'in-' === substr( $class, 0, 3 ) ) {
			$where = 'in';
		}
		if ( 'on-' === substr( $class, 0, 3 ) ) {
			$where = 'on';
		}
		if ( 'has-' === substr( $class, 0, 4 ) ) {
			$where = 'has';
		}
		if ( 'by-' === substr( $class, 0, 3 ) ) {
			$where = 'by';
		}
		if ( 'with-' === substr( $class, 0, 5 ) ) {
			$where = 'with';
		}
		if ( 'style-' === substr( $class, 0, 5 ) ) {
			$where = 'style';
		}

		$groups[ $where ][] = $class;
	}
	sort( $groups['def'] );
	sort( $groups['is'] );
	sort( $groups['in'] );
	sort( $groups['on'] );
	sort( $groups['by'] );
	sort( $groups['has'] );
	sort( $groups['with'] );
	sort( $groups['other'] );
	return array_merge(
		$groups['other'],
		$groups['is'],
		$groups['in'],
		$groups['on'],
		$groups['by'],
		$groups['has'],
		$groups['with']
	);
}
