<?php
/**
 * Simplify admin for a more streamlined WordPress experience
 *
 * @package KDWeb\Plugin\Admin
 */

namespace KDWeb\Plugin\Admin;

if ( ! defined( 'ABSPATH' ) ) { exit; }

/**
 * Remove admin bar nodes
 *
 * @return void
 */
function simplify_admin_bar_nodes() {
	global $wp_admin_bar;
	$wp_admin_bar->remove_node( 'comments' );
	$wp_admin_bar->remove_node( 'customize' );
	$wp_admin_bar->remove_node( 'gravityforms-new-form' );
	$wp_admin_bar->remove_node( 'gform-forms' );
	$wp_admin_bar->remove_node( 'new-user' );
	$wp_admin_bar->remove_node( 'themes' );
	$wp_admin_bar->remove_node( 'updates' );
	$wp_admin_bar->remove_node( 'user' );
	$wp_admin_bar->remove_node( 'wp-logo' );
	$wp_admin_bar->remove_node( 'wpseo-menu' );

	remove_action( 'wp_before_admin_bar_render', array( 'GFForms', 'admin_bar' ) );
}

/**
 * Remove admin dashboard meta boxes
 *
 * @return void
 */
function simplify_admin_dashboard() {
	global $wp_meta_boxes;
	remove_action( 'welcome_panel', 'wp_welcome_panel' );
	remove_meta_box( 'dashboard_quick_press', 'dashboard', 'side' );
	remove_meta_box( 'dashboard_quick_press', 'dashboard', 'side' );
	remove_meta_box( 'dashboard_primary', 'dashboard', 'side' );
	remove_meta_box( 'rg_forms_dashboard', 'dashboard', 'normal' );
	remove_meta_box( 'dashboard_site_health', 'dashboard', 'normal' );
	remove_meta_box( 'dashboard_right_now', 'dashboard', 'normal' );
	remove_meta_box( 'wpseo-dashboard-overview', 'dashboard', 'normal' );
	remove_meta_box( 'wpseo-wincher-dashboard-overview', 'dashboard', 'normal' );
}

/**
 * Disable moving checked terms to the top of the list in the admin
 *
 * @param array $args The term walker arguments.
 * @return array
 */
function disable_taxonomy_checked_ontop( $args ) {
	$args['checked_ontop'] = false;
	return $args;
}

/**
 * Set some basic image settings
 *
 * @return void
 */
function simplify_image_options() {
	update_option( 'image_default_size', apply_filters( 'kdweb_default_image_size', 'large' ) );
	update_option( 'image_default_align', apply_filters( 'kdweb_default_image_align', 'center' ) );
}

/**
 * Remove Appearance as a parent for Menus in the admin menu
 *
 * @return void
 */
function admin_menu_menus_adjust_parent() {
	global $pagenow, $parent_file;
	if ( 'nav-menus.php' === $pagenow && 'themes.php' === $parent_file ) {
		$parent_file = '';
	}
}

/**
 * Add menus to admin menu
 *
 * @return void
 */
function admin_menu_add_menus() {
	if ( ! current_theme_supports( 'menus' ) ) {
		return;
	}
	remove_submenu_page( 'themes.php', 'nav-menus.php' );
	add_menu_page(
		__( 'Menus', 'kdweb' ),
		__( 'Menus', 'kdweb' ),
		'edit_theme_options',
		'nav-menus.php',
		'',
		'dashicons-menu',
		26
	);
}

/**
 * Make the redirection menu item active for admin users when they are on that page
 */
function admin_menu_redirection_adjust_parent() {
	if ( ! defined( 'REDIRECTION_VERSION' ) || ! current_user_can( 'admin' ) || ! isset( $_GET['page'] ) || $_GET['page'] !== 'redirection.php' ) {
		return;
	}
	global $menu;
	foreach ( $menu as $index => $item ) {
		if ( 'tools.php?page=redirection.php' === $item[2] ) {
			$menu[ $index ][4] = ' menu-top current';
		}
	}
}

/**
 * Add redirection to the menu for admin users
 *
 * @return void
 */
function admin_menu_redirection() {
	if ( ! defined( 'REDIRECTION_VERSION' ) || ! current_user_can( 'admin' ) ) {
		return;
	}
	add_menu_page(
		__( 'Redirects', 'default' ),
		__( 'Redirects', 'default' ),
		'kdweb_access_admin',
		'tools.php?page=redirection.php',
		'',
		'dashicons-redo',
		110
	);
}

/**
 * Make the facet wp menu item active for admin users when they are on that page
 */
function admin_menu_facetwp_adjust_parent() {
	if ( ! defined( 'FACETWP_VERSION' ) || ! current_user_can( 'admin' ) || ! isset( $_GET['page'] ) || $_GET['page'] !== 'facetwp' ) {
		return;
	}
	global $menu;
	foreach ( $menu as $index => $item ) {
		if ( 'options-general.php?page=facetwp' === $item[2] ) {
			$menu[ $index ][4] = ' menu-top current';
		}
	}
}

/**
 * Add facet wp to the menu for admin users
 *
 * @return void
 */
function admin_menu_facetwp() {
	if ( ! defined( 'FACETWP_VERSION' ) || ! current_user_can( 'admin' ) ) {
		return;
	}
	add_menu_page(
		__( 'Filters', 'default' ),
		__( 'Filters', 'default' ),
		'kdweb_access_admin',
		'options-general.php?page=facetwp',
		'',
		'dashicons-filter',
		40
	);
}


/**
 * Make the relevanssi menu item active for admin users when they are on that page
 */
function admin_menu_relevanssi_adjust_parent() {
	if ( ! defined( 'RELEVANSSI_PREMIUM' ) || ! current_user_can( 'admin' ) || ! isset( $_GET['page'] ) || ! strstr( $_GET['page'], 'relevanssi' ) ) {
		return;
	}
	global $menu;
	foreach ( $menu as $index => $item ) {
		if ( 'options-general.php?page=relevanssi-premium/relevanssi.php' === $item[2] ) {
			$menu[ $index ][4] = ' menu-top current';
		}
	}
}

/**
 * Add facet wp to the menu for admin users
 *
 * @return void
 */
function admin_menu_relevanssi() {
	if ( ! defined( 'RELEVANSSI_PREMIUM' ) || ! current_user_can( 'admin' ) ) {
		return;
	}
	add_menu_page(
		__( 'Relevancy search', 'default' ),
		__( 'Relevancy search', 'default' ),
		'kdweb_access_admin',
		'options-general.php?page=relevanssi-premium/relevanssi.php',
		'',
		'dashicons-search',
		50
	);
}

/**
 * Remove themes from admin menu
 *
 * @return void
 */
function admin_menu_remove_themes() {
	global $submenu;
	$themes_useless = false;
	if ( isset( $submenu['themes.php'] ) && ! current_user_can( 'install_themes' ) ) {
		$has_menus = isset( $submenu['themes.php'][10] );
		if ( ! $has_menus || $has_menus && 2 === count( $submenu['themes.php'] ) ) {
			$all_themes = wp_get_themes();
			if ( 1 === count( $all_themes ) ) {
				$themes_useless = true;
			}
		}
	}
	if (
		! $themes_useless &&
		(
			current_user_can( 'delete_themes' ) ||
			current_user_can( 'edit_themes' ) ||
			current_user_can( 'install_themes' ) ||
			current_user_can( 'resume_themes' ) ||
			current_user_can( 'switch_themes' ) ||
			current_user_can( 'update_themes' )
		)
	) {
		return;
	}
	remove_menu_page( 'themes.php' );
}

/**
 * Remove tools from admin menu
 *
 * @return void
 */
function admin_menu_remove_tools() {
	if (
		current_user_can( 'import' ) ||
		current_user_can( 'export' ) ||
		current_user_can( 'view_site_health_checks' ) ||
		current_user_can( 'export_others_personal_data' ) ||
		current_user_can( 'erase_others_personal_data' ) ||
		current_user_can( 'manage_options' )
	) {
		return;
	}
	remove_menu_page( 'tools.php' );
}

/**
 * Remove settings from admin menu
 *
 * @return void
 */
function admin_menu_remove_settings() {
	if ( current_user_can( 'manage_options' ) ) {
		return;
	}
	remove_menu_page( 'options-general.php' );
}

/**
 * Remove settings from admin menu
 *
 * @return void
 */
function admin_menu_remove_updates() {
	if ( current_user_can( 'administrator' ) ) {
		return;
	}
	remove_submenu_page( 'index.php', 'update-core.php' );
}

/**
 * Remove settings from admin menu
 *
 * @return void
 */
function admin_menu_remove_acf() {
	if ( current_user_can( 'administrator' ) ) {
		return;
	}
	remove_menu_page( 'edit.php?post_type=acf-field-group' );
}

/**
 * Remove settings from admin menu
 *
 * @return void
 */
function admin_menu_remove_yoast_premium_pages() {
	if ( ! function_exists( 'YoastSEO' ) || YoastSEO()->helpers->product->is_premium() ) {
		return;
	}
	remove_menu_page( 'wpseo_page_academy' );
	remove_menu_page( 'wpseo_licenses' );
	remove_menu_page( 'wpseo_workouts' );
	remove_menu_page( 'wpseo_redirects' );
	remove_submenu_page( 'wpseo_dashboard', 'wpseo_page_academy' );
	remove_submenu_page( 'wpseo_dashboard', 'wpseo_licenses' );
	remove_submenu_page( 'wpseo_dashboard', 'wpseo_workouts' );
	remove_submenu_page( 'wpseo_dashboard', 'wpseo_redirects' );
}

/**
 * Fix privacy page link on the login page so that it gets removed if the privacy page does not exist or it has wrong settings
 *
 * @param string $link The html output for the anchor.
 * @return null|string
 */
function fix_login_privacy_policy_link( $link ) {
	global $pagenow;
	if ( 'wp-login.php' !== $pagenow ) {
		return $link;
	}
	$privacy_post = get_post( get_option( 'wp_page_for_privacy_policy' ) );
	if ( ! $privacy_post || 'publish' !== $privacy_post->post_status || 'page' !== $privacy_post->post_type || ! empty( $privacy_post->post_password ) ) {
		return null;
	}
	return $link;
}

/**
 * Check if post type has capability for single pages, and make sure the admin bar view links work properly according to it
 *
 * @param WP_Admin_Bar $wp_admin_bar The admin bar object.
 * @return void
 */
function fix_unqueriable_post_view_post_link_from_admin_bar( $wp_admin_bar ) {
	global $post_id;
	if ( ! is_admin() ) {
		return;
	}
	$screen           = get_current_screen();
	$post             = 'edit-comments' === $screen->base && ! empty( $post_id ) ? get_post( $post_id ) : get_post();
	$post_type_object = 'post' === $screen->base ? get_post_type_object( $post->post_type ) : ( 'edit' === $screen->base ? get_post_type_object( $screen->post_type ) : ( 'edit-comments' === $screen->base && ! empty( $post ) ? get_post_type_object( $post ) : null ) );
	if ( ! $post_type_object || ! is_admin_bar_showing() || 'add' === $screen->action || empty( $post ) || ! current_user_can( 'read_post', $post->ID ) ) {
		return;
	}
	$has_single = post_type_has_single( $post_type_object->name );
	$bar_nodes  = $wp_admin_bar->get_nodes();
	if ( $has_single && ! isset( $bar_nodes['view'] ) ) {
		if ( 'draft' === $post->post_status ) {
			$wp_admin_bar->add_node(
				array(
					'id'    => 'preview',
					'title' => $post_type_object->labels->view_item,
					'href'  => esc_url( get_preview_post_link( $post ) ),
					'meta'  => array( 'target' => "wp-preview-{$post->ID}" ),
				)
			);
		} else {
			$wp_admin_bar->add_node(
				array(
					'id'    => 'view',
					'title' => $post_type_object->labels->view_item,
					'href'  => get_permalink( $post->ID ),
				)
			);
		}
	}
	if ( ! $has_single && isset( $bar_nodes['view'] ) ) {
		$wp_admin_bar->remove_node( 'preview' );
		$wp_admin_bar->remove_node( 'view' );
	}
}

/**
 * Fix shortlinks for post types that do not have single pages
 *
 * @return void
 */
function fix_unqueriable_post_buttons() {
	global $post_id;
	if ( ! is_admin() ) {
		return;
	}
	$screen           = get_current_screen();
	$post             = 'edit-comments' === $screen->base && ! empty( $post_id ) ? get_post( $post_id ) : get_post();
	$post_type_object = 'post' === $screen->base ? get_post_type_object( $post->post_type ) : ( 'edit' === $screen->base ? get_post_type_object( $screen->post_type ) : ( 'edit-comments' === $screen->base && ! empty( $post ) ? get_post_type_object( $post ) : null ) );
	if ( ! $post_type_object || 'add' === $screen->action || empty( $post ) || ! current_user_can( 'read_post', $post->ID ) || post_type_has_single( $post_type_object->name ) ) {
		return;
	}
	?><style>#edit-slug-box,.edit-post-post-url,#preview-action{display:none!important}</style><?php
}

/**
 * Fix the row actions for unqueriable post types in the admin
 *
 * @param array $actions An array of row action links
 * @param WP_Post $post The post object.
 * @return array
 */
function fix_unqueriable_post_row_actions( $actions, $post ) {
	$screen           = get_current_screen();
	$post_type_object = 'post' === $screen->base ? get_post_type_object( $post->post_type ) : ( 'edit' === $screen->base ? get_post_type_object( $screen->post_type ) : ( 'edit-comments' === $screen->base && ! empty( $post ) ? get_post_type_object( $post ) : null ) );
	if ( ! $post_type_object || 'add' === $screen->action || empty( $post ) || ! current_user_can( 'read_post', $post->ID ) ) {
		return $actions;
	}
	$has_single = post_type_has_single( $post_type_object->name );
	if ( $has_single && ! isset( $actions['view'] ) ) {
		$actions['view'] = sprintf(
			'<a href="%s" rel="bookmark" aria-label="%s">%s</a>',
			esc_url( get_preview_post_link( $post ) ),
			esc_attr( sprintf( __( 'Preview &#8220;%s&#8221;' ), _draft_or_post_title() ) ),
			__( 'Preview' )
		);
	}
	if ( ! $has_single && isset( $actions['view'] ) ) {
		unset( $actions['view'] );
	}
	return $actions;
}

/**
 * Remove rewrite rule permastructs for post types that do not have single pages
 *
 * @param array $post_rewrite Array of rewrite rules for posts, keyed by their regex pattern.
 * @return array
 */
function fix_unqueriable_permalink_structures( $post_rewrite ) {
	foreach ( get_post_types( array( '_builtin' => false, 'public' => true, 'publicly_queryable' => false ) ) as $post_type ) {
		if ( ! post_type_has_single( $post_type ) ) {
			remove_permastruct( $post_type );
		}
	}
	return $post_rewrite;
}

/**
 * Make sure post types that do not have single pages are not viewable
 *
 * @param boolean $is_viewable Whether the post type is viewable.
 * @param WP_Post_Type $post_type_object The post type object.
 * @return boolean
 */
function fix_unqueriable_post_type_viewable( $is_viewable, $post_type_object ) {
	$has_single = post_type_has_single( $post_type_object->name );
	return $has_single ? $is_viewable : false;
}

/**
 * Fix post type queries that do not have single pages but have archives
 *
 * @param WP_Query $wp_query The main query object.
 * @return void
 */
function fix_unqueriable_post_type_queries( $wp_query ) {
	global $wp;
	if ( is_admin() || ! $wp_query->is_main_query() || ! str_starts_with( $wp->matched_query, 'post_type=' ) || isset( $wp_query->query['post_type'] ) ) {
		return;
	}
	$parsed_query = wp_parse_args( $wp->matched_query );
	if ( ! isset( $parsed_query['post_type'] ) ) {
		return;
	}
	$has_archive = post_type_has_archive( $parsed_query['post_type'] );
	if ( ! $has_archive ) {
		return;
	}
	$wp_query->set( 'post_type', $parsed_query['post_type'] );
	$wp_query->is_post_type_archive = true;
}

/**
 * Remove admin dashboard meta boxes
 *
 * @return void
 */
function overwrite_admin_footer_thankyou() {
	echo apply_filters( 'kdweb_admin_footer_thankyou', '<span id="footer-thankyou">Developed by <a href="https://www.kdweb.co.uk" target="_blank">KD Web</a></span>' );
}

/**
 * Point login page logo to home instead of wordpress.org
 *
 * @return void
 */
function login_logo_to_home() {
	return home_url();
}

/**
 * Filter the next, previous and submit buttons for Gravity Forms and make them <button> elements
 *
 * @param string $button The button html.
 * @return string
 */
function gf_submit_to_button( $button ) {
	$fragment = \WP_HTML_Processor::create_fragment( $button );
    $fragment->next_token();
    $attributes = array( 'id', 'type', 'class', 'onclick' );
    $new_attributes = array();
    foreach ( $attributes as $attribute ) {
        $value = $fragment->get_attribute( $attribute );
        if ( ! empty( $value ) ) {
            $new_attributes[] = sprintf( '%s="%s"', $attribute, esc_attr( $value ) );
        }
    }

    return sprintf( '<button %s>%s</button>', implode( ' ', $new_attributes ), esc_html( $fragment->get_attribute( 'value' ) ) );
}

/**
 * Move Yoast Meta Box below ACF fields
 *
 * @return void
 */
function lower_yoast_metabox() {
	return 'low';
}

/**
 * Move Yoast Meta Box to the sidebar in wp editor
 *
 * @return void
 */
function move_yoast_metabox_to_sidebar() {
	global $wp_meta_boxes;
	if ( ! class_exists( 'WPSEO_Post_Type' ) ) {
		return;
	}
	$post_types = \WPSEO_Post_Type::get_accessible_post_types();
	foreach ( $post_types as $post_type ) {
		if ( isset( $wp_meta_boxes[ $post_type ]['normal']['low']['wpseo_meta'] ) ) {
			$wp_meta_boxes[ $post_type ]['side']['low']['wpseo_meta'] = $wp_meta_boxes[ $post_type ]['normal']['low']['wpseo_meta'];
			unset( $wp_meta_boxes[ $post_type ]['normal']['low']['wpseo_meta'] );
		}
	}
}

/**
 * Change the default settings for the post edit metaboxes
 *
 * @return void
 */
function change_default_post_edit_metaboxes() {
	return apply_filters(
		'kdweb_hide_post_edit_metaboxes',
		array(
			'authordiv',
			'commentstatusdiv',
			'commentsdiv',
			'postcustom',
			'revisionsdiv',
			'slugdiv',
			'tagsdiv-post_tag',
			'trackbacksdiv',
		)
	);
}

/**
 * Hide the excerpt editor description that leads admin user out of website
 *
 * @return void
 */
function hide_excerpt_description() {
	$description = __( 'A summary or blurb of the content. Used for archive pages and feeds.', 'kdweb' );
	?><style>#postexcerpt .inside p,.components-panel__body .editor-post-excerpt .components-external-link{display:none}#postexcerpt .inside::after,.components-panel__body .editor-post-excerpt::after{display:block;content:"<?php echo $description; ?>";margin:1em 0 0 0}</style><?php
}

/**
 * Hide commonly used plugin distrations
 *
 * @return void
 */
function declutter_plugin_pages() {
	wp_enqueue_style( 'setup-tinymce-fix-toolbar-styles', plugin_dir_url( dirname( __FILE__ ) ) . 'assets/declutter-plugin-pages.css', array(), '1.0.0' );
}

/**
 * Remove ACF admin footer text
 *
 * @return void
 */
function remove_acf_admin_footer_hook() {
	if ( ! function_exists( 'acf_get_instance' ) ) {
		return;
	}
	remove_filter( 'admin_footer_text', array( acf_get_instance( 'ACF_Admin' ), 'admin_footer_text' ), 10 );
	remove_filter( 'update_footer', array( acf_get_instance( 'ACF_Admin' ), 'admin_footer_version_text' ), 10 );
}

/**
 * Fix the Yoast SEO ID for admin pages
 *
 * @param int $id The post ID.
 * @return int
 */
function admin_yoast_simple_page_id( $id ) {
	global $post;
	if ( ! is_admin() || ! $post ) {
		return $id;
	}
	return $post->ID;
}

/**
 * Get a default set of  filters for meta data.
 *
 * @param array $exclusions The meta keys to exclude (uses str_starts_with).
 * @return array
 */
function export_default_meta_exclusions( $exclusions ) {
	$exclusions = array_merge(
		$exclusions,
		array(
			'_edit_lock',
			'_edit_last',
			'_mtsnb_override_bar',
			'_oembed_',
			'_wp_attached_file',
			'_wp_attachment_metadata',
			'_yoast_wpseo_content_score',
			'_yoast_wpseo_estimated-reading-time-minutes',
			'_yoast_wpseo_primary_group',
			'_yoast_wpseo_primary_practisearea',
			'_yoast_wpseo_wordproof_timestamp',
			'additional_property_',
			'ao_post_optimize',
			'apartment_',
			'article_items',
			'blogposting',
			'faq_',
			'feed_',
			'howto_',
			'image_ob',
			'itemlist_',
			'local_enable_',
			'newsarticle_',
			'mc_',
			'music_',
			'movie_',
			'performer_',
			'product_',
			'saswp_',
			'tech_',
			'trip_',
			'tvseries_',
		)
	);
	return $exclusions;
}

/**
 * Update the author of the post when importing with the original user.
 *
 * @param array $data The already filtered post array to import.
 * @param array $post The post array to import.
 * @return array
 */
function replace_author_when_importing_posts( $data, $post ) {
	$author = get_user_id_from_original( $post['post_author'] );
	if ( $author ) {
		$data['post_author'] = $author;
	}
	return $data;
}

/**
 * Update the post parent of the post when importing with the original user.
 *
 * @param array $data The already filtered post array to import.
 * @param array $post The post array to import.
 * @return array
 */
function replace_parent_when_importing_posts( $data, $post ) {
	$parent = get_post_id_from_original( $post['post_parent'] );
	if ( $parent ) {
		$data['post_parent'] = $parent;
	}
	return $data;
}

/**
 * Check if post type items have single pages
 *
 * @param string $post_type The post type name.
 * @return bool
 */
function post_type_has_single( $post_type = 'post' ) {
	if ( 'page' === $post_type ) {
		return true;
	}
	$post_type_object = get_post_type_object( $post_type );
	if ( ! $post_type_object ) {
		return false;
	}
	return true === $post_type_object->publicly_queryable && ! empty( $post_type_object->query_var );
}
