<?php
/**
 * Admin Settings Page
 *
 * @package KDWeb\Plugin\Admin
 */

namespace KDWeb\Plugin\Admin;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Get the primary color from the active theme's theme.json file.
 *
 * @return string The primary color hex code, or empty string if not found.
 */
function get_theme_primary_color() {
	$kdweb_json_path = get_stylesheet_directory() . '/kdweb.json';

	if ( ! file_exists( $kdweb_json_path ) ) {
		return '';
		}

	$kdweb_json_content = file_get_contents( $kdweb_json_path );
	if ( false === $kdweb_json_content ) {
		return '';
	}

	$kdweb_json = json_decode( $kdweb_json_content, true );
	if ( ! is_array( $kdweb_json ) ) {
		return '';
	}

	// Check for settings.color.palette array
	if ( isset( $kdweb_json['palette'] ) && is_array( $kdweb_json['palette'] ) ) {
		foreach ( $kdweb_json['palette'] as $slug => $color ) {
			if ( 'primary' === $slug ) {
				return $color;
			}
		}
	}

	return '';
}

/**
 * Get default settings for the plugin.
 *
 * @return array Array of default settings.
 */
function get_default_settings() {
	return array(
		// Enabled features
		'kdweb_admin_enabled_features' => array(
			'create_admin_role',
			'disable_asset_versioning',
			'disable_auto_updates',
			'disable_custom_css',
			'disable_customizer',
			'disable_feeds',
			'disable_json_api',
			'disable_xmlprc',
			'extend_sticky_posts',
			'hide_blocks',
			'setup_customized_login_page',
			'setup_dashboard_info',
			'setup_dashboard_launchpad',
			'simplify_admin',
			'simplify_html_output',
		),
		// Undeletable Content
		'kdweb_admin_undeletable_key_pages' => true,
		'kdweb_admin_undeletable_posts' => array(),
		'kdweb_admin_undeletable_terms' => array(),
		// Hide Blocks
		'kdweb_admin_hidden_blocks' => array(
			'core/accordion', 'core/accordion-heading', 'core/accordion-item', 'core/accordion-panel',
			'core/archives', 'core/avatar', 'core/block', 'core/button', 'core/buttons', 'core/calendar',
			'core/categories', 'core/code', 'core/column', 'core/columns', 'core/comment-author-name',
			'core/comment-content', 'core/comment-date', 'core/comment-edit-link', 'core/comment-reply-link',
			'core/comment-template', 'core/comments', 'core/comments-pagination', 'core/comments-pagination-next',
			'core/comments-pagination-numbers', 'core/comments-pagination-previous', 'core/comments-title',
			'core/cover', 'core/footnotes', 'core/freeform', 'core/gallery', 'core/group', 'core/home-link',
			'core/html', 'core/latest-comments', 'core/latest-posts', 'core/legacy-widget', 'core/loginout',
			'core/math', 'core/media-text', 'core/missing', 'core/more', 'core/navigation', 'core/navigation-link',
			'core/navigation-submenu', 'core/nextpage', 'core/page-list', 'core/page-list-item', 'core/post-author',
			'core/post-author-biography', 'core/post-author-name', 'core/post-comments', 'core/post-comments-count',
			'core/post-comments-form', 'core/post-comments-link', 'core/post-content', 'core/post-date',
			'core/post-excerpt', 'core/post-featured-image', 'core/post-navigation-link', 'core/post-template',
			'core/post-terms', 'core/post-time-to-read', 'core/post-title', 'core/preformatted', 'core/pullquote',
			'core/query', 'core/query-no-results', 'core/query-pagination', 'core/query-pagination-next',
			'core/query-pagination-numbers', 'core/query-pagination-previous', 'core/query-title', 'core/query-total',
			'core/quote', 'core/read-more', 'core/rss', 'core/search', 'core/separator', 'core/shortcode',
			'core/site-logo', 'core/site-tagline', 'core/site-title', 'core/social-link', 'core/social-links',
			'core/spacer', 'core/tag-cloud', 'core/template-part', 'core/term-count', 'core/term-description',
			'core/term-name', 'core/term-template', 'core/terms-query', 'core/text-columns', 'core/verse',
			'core/widget-group', 'gravityforms/form', 'woocommerce/accordion-group', 'woocommerce/accordion-header',
			'woocommerce/accordion-item', 'woocommerce/accordion-panel', 'woocommerce/active-filters',
			'woocommerce/add-to-cart-form', 'woocommerce/add-to-cart-with-options',
			'woocommerce/add-to-cart-with-options-grouped-product-item',
			'woocommerce/add-to-cart-with-options-grouped-product-item-label',
			'woocommerce/add-to-cart-with-options-grouped-product-item-selector',
			'woocommerce/add-to-cart-with-options-grouped-product-selector',
			'woocommerce/add-to-cart-with-options-quantity-selector',
			'woocommerce/add-to-cart-with-options-variation-description',
			'woocommerce/add-to-cart-with-options-variation-selector',
			'woocommerce/add-to-cart-with-options-variation-selector-attribute',
			'woocommerce/add-to-cart-with-options-variation-selector-attribute-name',
			'woocommerce/add-to-cart-with-options-variation-selector-attribute-options',
			'woocommerce/all-products', 'woocommerce/all-reviews', 'woocommerce/attribute-filter',
			'woocommerce/breadcrumbs', 'woocommerce/cart', 'woocommerce/cart-accepted-payment-methods-block',
			'woocommerce/cart-cross-sells-block', 'woocommerce/cart-cross-sells-products-block',
			'woocommerce/cart-express-payment-block', 'woocommerce/cart-items-block',
			'woocommerce/cart-line-items-block', 'woocommerce/cart-link', 'woocommerce/cart-order-summary-block',
			'woocommerce/cart-order-summary-coupon-form-block', 'woocommerce/cart-order-summary-discount-block',
			'woocommerce/cart-order-summary-fee-block', 'woocommerce/cart-order-summary-heading-block',
			'woocommerce/cart-order-summary-shipping-block', 'woocommerce/cart-order-summary-subtotal-block',
			'woocommerce/cart-order-summary-taxes-block', 'woocommerce/cart-order-summary-totals-block',
			'woocommerce/cart-totals-block', 'woocommerce/catalog-sorting', 'woocommerce/category-description',
			'woocommerce/category-title', 'woocommerce/checkout', 'woocommerce/checkout-actions-block',
			'woocommerce/checkout-additional-information-block', 'woocommerce/checkout-billing-address-block',
			'woocommerce/checkout-contact-information-block', 'woocommerce/checkout-express-payment-block',
			'woocommerce/checkout-fields-block', 'woocommerce/checkout-order-note-block',
			'woocommerce/checkout-order-summary-block', 'woocommerce/checkout-order-summary-cart-items-block',
			'woocommerce/checkout-order-summary-coupon-form-block', 'woocommerce/checkout-order-summary-discount-block',
			'woocommerce/checkout-order-summary-fee-block', 'woocommerce/checkout-order-summary-shipping-block',
			'woocommerce/checkout-order-summary-subtotal-block', 'woocommerce/checkout-order-summary-taxes-block',
			'woocommerce/checkout-order-summary-totals-block', 'woocommerce/checkout-payment-block',
			'woocommerce/checkout-pickup-options-block', 'woocommerce/checkout-shipping-address-block',
			'woocommerce/checkout-shipping-method-block', 'woocommerce/checkout-shipping-methods-block',
			'woocommerce/checkout-terms-block', 'woocommerce/checkout-totals-block', 'woocommerce/classic-shortcode',
			'woocommerce/coming-soon', 'woocommerce/customer-account', 'woocommerce/email-content',
			'woocommerce/empty-cart-block', 'woocommerce/empty-mini-cart-contents-block',
			'woocommerce/featured-category', 'woocommerce/featured-product', 'woocommerce/filled-cart-block',
			'woocommerce/filled-mini-cart-contents-block', 'woocommerce/filter-wrapper',
			'woocommerce/handpicked-products', 'woocommerce/legacy-template', 'woocommerce/mini-cart',
			'woocommerce/mini-cart-cart-button-block', 'woocommerce/mini-cart-checkout-button-block',
			'woocommerce/mini-cart-contents', 'woocommerce/mini-cart-footer-block', 'woocommerce/mini-cart-items-block',
			'woocommerce/mini-cart-products-table-block', 'woocommerce/mini-cart-shopping-button-block',
			'woocommerce/mini-cart-title-block', 'woocommerce/mini-cart-title-items-counter-block',
			'woocommerce/mini-cart-title-label-block', 'woocommerce/order-confirmation-additional-fields',
			'woocommerce/order-confirmation-additional-fields-wrapper',
			'woocommerce/order-confirmation-additional-information', 'woocommerce/order-confirmation-billing-address',
			'woocommerce/order-confirmation-billing-wrapper', 'woocommerce/order-confirmation-create-account',
			'woocommerce/order-confirmation-downloads', 'woocommerce/order-confirmation-downloads-wrapper',
			'woocommerce/order-confirmation-shipping-address', 'woocommerce/order-confirmation-shipping-wrapper',
			'woocommerce/order-confirmation-status', 'woocommerce/order-confirmation-summary',
			'woocommerce/order-confirmation-totals', 'woocommerce/order-confirmation-totals-wrapper',
			'woocommerce/page-content-wrapper', 'woocommerce/payment-method-icons', 'woocommerce/price-filter',
			'woocommerce/proceed-to-checkout-block', 'woocommerce/product-average-rating',
			'woocommerce/product-best-sellers', 'woocommerce/product-button', 'woocommerce/product-categories',
			'woocommerce/product-category', 'woocommerce/product-collection', 'woocommerce/product-collection-no-results',
			'woocommerce/product-description', 'woocommerce/product-details', 'woocommerce/product-filter-active',
			'woocommerce/product-filter-attribute', 'woocommerce/product-filter-checkbox-list',
			'woocommerce/product-filter-chips', 'woocommerce/product-filter-clear-button',
			'woocommerce/product-filter-price', 'woocommerce/product-filter-price-slider',
			'woocommerce/product-filter-rating', 'woocommerce/product-filter-removable-chips',
			'woocommerce/product-filter-status', 'woocommerce/product-filter-taxonomy', 'woocommerce/product-filters',
			'woocommerce/product-gallery', 'woocommerce/product-gallery-large-image',
			'woocommerce/product-gallery-large-image-next-previous', 'woocommerce/product-gallery-thumbnails',
			'woocommerce/product-image', 'woocommerce/product-image-gallery', 'woocommerce/product-meta',
			'woocommerce/product-new', 'woocommerce/product-on-sale', 'woocommerce/product-price',
			'woocommerce/product-query', 'woocommerce/product-rating', 'woocommerce/product-rating-counter',
			'woocommerce/product-rating-stars', 'woocommerce/product-results-count',
			'woocommerce/product-review-author-name', 'woocommerce/product-review-content',
			'woocommerce/product-review-date', 'woocommerce/product-review-form', 'woocommerce/product-review-rating',
			'woocommerce/product-review-template', 'woocommerce/product-reviews',
			'woocommerce/product-reviews-pagination', 'woocommerce/product-reviews-pagination-next',
			'woocommerce/product-reviews-pagination-numbers', 'woocommerce/product-reviews-pagination-previous',
			'woocommerce/product-reviews-title', 'woocommerce/product-sale-badge', 'woocommerce/product-search',
			'woocommerce/product-sku', 'woocommerce/product-specifications', 'woocommerce/product-stock-indicator',
			'woocommerce/product-summary', 'woocommerce/product-tag', 'woocommerce/product-template',
			'woocommerce/product-title', 'woocommerce/product-top-rated', 'woocommerce/products-by-attribute',
			'woocommerce/rating-filter', 'woocommerce/related-products', 'woocommerce/reviews-by-category',
			'woocommerce/reviews-by-product', 'woocommerce/single-product', 'woocommerce/stock-filter',
			'woocommerce/store-notices',
		),
		'kdweb_admin_hide_blocks_show_all_to_admin' => false,
	);
}

/**
 * Add a settings page to the admin settings menu.
 *
 * @return void
 */
function add_settings_page() {
	add_options_page(
		'KD Web Settings',
		'KD Web',
		'manage_options',
		'kdweb-admin-settings',
		'KDWeb\Plugin\Admin\render_settings_page'
	);
}
add_action( 'admin_menu', 'KDWeb\Plugin\Admin\add_settings_page' );

/**
 * AJAX handler for searching posts
 *
 * @return void
 */
function ajax_search_posts() {
	check_ajax_referer( 'kdweb_admin_search', 'nonce' );

	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( 'Insufficient permissions' );
	}

	$search = isset( $_GET['q'] ) ? sanitize_text_field( wp_unslash( $_GET['q'] ) ) : '';
	$page = isset( $_GET['page'] ) ? absint( $_GET['page'] ) : 1;

	$args = array(
		'post_type'      => 'any',
		'post_status'    => 'any',
		'posts_per_page' => 20,
		'paged'          => $page,
		'orderby'        => 'title',
		'order'          => 'ASC',
	);

	if ( ! empty( $search ) ) {
		$args['s'] = $search;
	}

	$query = new \WP_Query( $args );
	$results = array();

	if ( $query->have_posts() ) {
		foreach ( $query->posts as $post ) {
			$results[] = array(
				'id'   => $post->ID,
				'text' => $post->post_title . ' (ID: ' . $post->ID . ', Type: ' . $post->post_type . ')',
			);
		}
	}

	wp_send_json( array(
		'results'    => $results,
		'pagination' => array(
			'more' => $query->max_num_pages > $page,
		),
	) );
}
add_action( 'wp_ajax_kdweb_search_posts', 'KDWeb\Plugin\Admin\ajax_search_posts' );

/**
 * AJAX handler for searching terms
 *
 * @return void
 */
function ajax_search_terms() {
	check_ajax_referer( 'kdweb_admin_search', 'nonce' );

	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( 'Insufficient permissions' );
	}

	$search = isset( $_GET['q'] ) ? sanitize_text_field( wp_unslash( $_GET['q'] ) ) : '';
	$page = isset( $_GET['page'] ) ? absint( $_GET['page'] ) : 1;
	$per_page = 20;
	$offset = ( $page - 1 ) * $per_page;

	$args = array(
		'taxonomy'   => get_taxonomies( array( 'public' => true ), 'names' ),
		'hide_empty' => false,
		'number'     => $per_page,
		'offset'     => $offset,
		'orderby'    => 'name',
		'order'      => 'ASC',
	);

	if ( ! empty( $search ) ) {
		$args['search'] = $search;
	}

	$terms = get_terms( $args );
	$results = array();

	if ( ! is_wp_error( $terms ) && ! empty( $terms ) ) {
		foreach ( $terms as $term ) {
			$results[] = array(
				'id'   => $term->term_id,
				'text' => $term->name . ' (ID: ' . $term->term_id . ', Taxonomy: ' . $term->taxonomy . ')',
			);
		}
	}

	// Check if there are more results
	$count_args = $args;
	unset( $count_args['number'], $count_args['offset'] );
	$total_terms = wp_count_terms( $count_args );
	$has_more = is_numeric( $total_terms ) && $total_terms > ( $offset + $per_page );

	wp_send_json( array(
		'results'    => $results,
		'pagination' => array(
			'more' => $has_more,
		),
	) );
}
add_action( 'wp_ajax_kdweb_search_terms', 'KDWeb\Plugin\Admin\ajax_search_terms' );

/** * AJAX handler for resetting settings to defaults
 *
 * @return void
 */
function ajax_reset_settings() {
	check_ajax_referer( 'kdweb_admin_reset', 'nonce' );

	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( 'Insufficient permissions' );
	}

	$defaults = get_default_settings();
	foreach ( $defaults as $option_name => $default_value ) {
		update_option( $option_name, $default_value );
	}

	wp_send_json_success( 'Settings have been reset to defaults.' );
}
add_action( 'wp_ajax_kdweb_reset_settings', 'KDWeb\\Plugin\\Admin\\ajax_reset_settings' );

/** * Display the settings page.
 *
 * @return void
 */
function render_settings_page() {
	if ( ! current_user_can( 'manage_options' ) ) {
		return;
	}

	// Enqueue WordPress media uploader
	wp_enqueue_media();

	// Enqueue WordPress color picker
	wp_enqueue_style( 'wp-color-picker' );
	wp_enqueue_script( 'wp-color-picker' );

	// Enqueue Select2
	wp_enqueue_style( 'select2', 'https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css', array(), '4.1.0' );
	wp_enqueue_script( 'select2', 'https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js', array( 'jquery' ), '4.1.0', true );

	// Check if settings were just reset
	if ( isset( $_GET['settings-reset'] ) && '1' === $_GET['settings-reset'] ) {
		echo '<div class="notice notice-success is-dismissible"><p>Settings have been reset to defaults.</p></div>';
	}

	if ( isset( $_POST['kdweb_admin_settings_nonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['kdweb_admin_settings_nonce'] ) ), 'kdweb_admin_settings_save' ) ) {
		// Save all feature toggles
		$features = array(
			'create_admin_role',
			'disable_asset_versioning',
			'disable_auto_updates',
			'disable_custom_css',
			'disable_customizer',
			'disable_feeds',
			'disable_json_api',
			'disable_xmlprc',
			'extend_sticky_posts',
			'hide_blocks',
			'setup_customized_login_page',
			'setup_dashboard_info',
			'setup_dashboard_launchpad',
			'simplify_admin',
			'simplify_html_output',
		);

		$enabled_features = array();
		foreach ( $features as $feature ) {
			if ( isset( $_POST[ 'kdweb_admin_' . $feature ] ) ) {
				$enabled_features[] = $feature;
			}
		}
		update_option( 'kdweb_admin_enabled_features', $enabled_features );

		// Save feature-specific settings
		// Disable Asset Versioning
		update_option( 'kdweb_admin_disable_asset_versioning_only_unset', isset( $_POST['kdweb_admin_disable_asset_versioning_only_unset'] ) );

		// Disable Auto Updates
		update_option( 'kdweb_admin_disable_auto_updates_core', isset( $_POST['kdweb_admin_disable_auto_updates_core'] ) );
		update_option( 'kdweb_admin_disable_auto_updates_minor', isset( $_POST['kdweb_admin_disable_auto_updates_minor'] ) );
		update_option( 'kdweb_admin_disable_auto_updates_plugin', isset( $_POST['kdweb_admin_disable_auto_updates_plugin'] ) );
		update_option( 'kdweb_admin_disable_auto_updates_theme', isset( $_POST['kdweb_admin_disable_auto_updates_theme'] ) );
		update_option( 'kdweb_admin_disable_auto_updates_translation', isset( $_POST['kdweb_admin_disable_auto_updates_translation'] ) );

		// Disable JSON API
		update_option( 'kdweb_admin_disable_json_api_for_unauthenticated_only', isset( $_POST['kdweb_admin_disable_json_api_for_unauthenticated_only'] ) );

		// Extend Sticky Posts
		update_option( 'kdweb_admin_extend_sticky_posts_author', isset( $_POST['kdweb_admin_extend_sticky_posts_author'] ) );
		update_option( 'kdweb_admin_extend_sticky_posts_custom', isset( $_POST['kdweb_admin_extend_sticky_posts_custom'] ) );
		update_option( 'kdweb_admin_extend_sticky_posts_date', isset( $_POST['kdweb_admin_extend_sticky_posts_date'] ) );
		update_option( 'kdweb_admin_extend_sticky_posts_feed', isset( $_POST['kdweb_admin_extend_sticky_posts_feed'] ) );
		update_option( 'kdweb_admin_extend_sticky_posts_post_type', isset( $_POST['kdweb_admin_extend_sticky_posts_post_type'] ) );
		update_option( 'kdweb_admin_extend_sticky_posts_posts_page', isset( $_POST['kdweb_admin_extend_sticky_posts_posts_page'] ) );
		update_option( 'kdweb_admin_extend_sticky_posts_taxonomy', isset( $_POST['kdweb_admin_extend_sticky_posts_taxonomy'] ) );

		// Logo & Icon Settings
		update_option( 'kdweb-site-logo', sanitize_text_field( wp_unslash( $_POST['kdweb_site_logo'] ?? '' ) ) );
		update_option( 'kdweb-site-icon', sanitize_text_field( wp_unslash( $_POST['kdweb_site_icon'] ?? '' ) ) );
		// Sync site icon with WordPress core option
		if ( isset( $_POST['kdweb_site_icon'] ) ) {
			update_option( 'site_icon', sanitize_text_field( wp_unslash( $_POST['kdweb_site_icon'] ) ) );
		}

		// Customized Login Page
		update_option( 'kdweb_admin_setup_customized_login_page_logo', isset( $_POST['kdweb_admin_setup_customized_login_page_logo'] ) );
		update_option( 'kdweb_admin_setup_customized_login_page_color', sanitize_text_field( wp_unslash( $_POST['kdweb_admin_setup_customized_login_page_color'] ?? '' ) ) );

		// Simplify Admin
		update_option( 'kdweb_admin_simplify_admin_admin_bar', isset( $_POST['kdweb_admin_simplify_admin_admin_bar'] ) );
		update_option( 'kdweb_admin_simplify_admin_dashboard', isset( $_POST['kdweb_admin_simplify_admin_dashboard'] ) );
		update_option( 'kdweb_admin_simplify_admin_taxonomy_checked_ontop', isset( $_POST['kdweb_admin_simplify_admin_taxonomy_checked_ontop'] ) );
		update_option( 'kdweb_admin_simplify_admin_image_options', isset( $_POST['kdweb_admin_simplify_admin_image_options'] ) );
		update_option( 'kdweb_admin_simplify_admin_menus', isset( $_POST['kdweb_admin_simplify_admin_menus'] ) );
		update_option( 'kdweb_admin_simplify_admin_redirection', isset( $_POST['kdweb_admin_simplify_admin_redirection'] ) );
		update_option( 'kdweb_admin_simplify_admin_facetwp', isset( $_POST['kdweb_admin_simplify_admin_facetwp'] ) );
		update_option( 'kdweb_admin_simplify_admin_relevanssi', isset( $_POST['kdweb_admin_simplify_admin_relevanssi'] ) );
		update_option( 'kdweb_admin_simplify_admin_hide_themes', isset( $_POST['kdweb_admin_simplify_admin_hide_themes'] ) );
		update_option( 'kdweb_admin_simplify_admin_hide_tools', isset( $_POST['kdweb_admin_simplify_admin_hide_tools'] ) );
		update_option( 'kdweb_admin_simplify_admin_hide_settings', isset( $_POST['kdweb_admin_simplify_admin_hide_settings'] ) );
		update_option( 'kdweb_admin_simplify_admin_hide_updates', isset( $_POST['kdweb_admin_simplify_admin_hide_updates'] ) );
		update_option( 'kdweb_admin_simplify_admin_hide_acf', isset( $_POST['kdweb_admin_simplify_admin_hide_acf'] ) );
		update_option( 'kdweb_admin_simplify_admin_hide_yoast_premium_pages', isset( $_POST['kdweb_admin_simplify_admin_hide_yoast_premium_pages'] ) );
		update_option( 'kdweb_admin_simplify_admin_fix_unqueriable_post_types', isset( $_POST['kdweb_admin_simplify_admin_fix_unqueriable_post_types'] ) );
		update_option( 'kdweb_admin_simplify_admin_gf_submit_to_button', isset( $_POST['kdweb_admin_simplify_admin_gf_submit_to_button'] ) );

		// Simplify HTML Output
		update_option( 'kdweb_admin_simplify_html_output_simple_mode', isset( $_POST['kdweb_admin_simplify_html_output_simple_mode'] ) );
		update_option( 'kdweb_admin_simplify_html_output_archive_title', isset( $_POST['kdweb_admin_simplify_html_output_archive_title'] ) );
		update_option( 'kdweb_admin_simplify_html_output_archive_link', isset( $_POST['kdweb_admin_simplify_html_output_archive_link'] ) );
		update_option( 'kdweb_admin_simplify_html_output_image_attributes', isset( $_POST['kdweb_admin_simplify_html_output_image_attributes'] ) );
		update_option( 'kdweb_admin_simplify_html_output_comment_form', isset( $_POST['kdweb_admin_simplify_html_output_comment_form'] ) );
		update_option( 'kdweb_admin_simplify_html_output_comment_form_website', isset( $_POST['kdweb_admin_simplify_html_output_comment_form_website'] ) );
		update_option( 'kdweb_admin_simplify_html_output_head_charset', isset( $_POST['kdweb_admin_simplify_html_output_head_charset'] ) );
		update_option( 'kdweb_admin_simplify_html_output_head_xua_compatible', isset( $_POST['kdweb_admin_simplify_html_output_head_xua_compatible'] ) );
		update_option( 'kdweb_admin_simplify_html_output_head_viewport', isset( $_POST['kdweb_admin_simplify_html_output_head_viewport'] ) );
		update_option( 'kdweb_admin_simplify_html_output_head_revisit_after', isset( $_POST['kdweb_admin_simplify_html_output_head_revisit_after'] ) );
		update_option( 'kdweb_admin_simplify_html_output_head_distribution', isset( $_POST['kdweb_admin_simplify_html_output_head_distribution'] ) );
		update_option( 'kdweb_admin_simplify_html_output_head_wp_version', isset( $_POST['kdweb_admin_simplify_html_output_head_wp_version'] ) );
		update_option( 'kdweb_admin_simplify_html_output_head_wlwmanifest_link', isset( $_POST['kdweb_admin_simplify_html_output_head_wlwmanifest_link'] ) );
		update_option( 'kdweb_admin_simplify_html_output_head_rsd_link', isset( $_POST['kdweb_admin_simplify_html_output_head_rsd_link'] ) );
		update_option( 'kdweb_admin_simplify_html_output_head_shortlink', isset( $_POST['kdweb_admin_simplify_html_output_head_shortlink'] ) );
		update_option( 'kdweb_admin_simplify_html_output_head_adjacent_posts_link', isset( $_POST['kdweb_admin_simplify_html_output_head_adjacent_posts_link'] ) );
		update_option( 'kdweb_admin_simplify_html_output_head_emoji', isset( $_POST['kdweb_admin_simplify_html_output_head_emoji'] ) );
		update_option( 'kdweb_admin_simplify_html_output_head_oembed', isset( $_POST['kdweb_admin_simplify_html_output_head_oembed'] ) );
		update_option( 'kdweb_admin_simplify_html_output_head_post_relational_links', isset( $_POST['kdweb_admin_simplify_html_output_head_post_relational_links'] ) );
		update_option( 'kdweb_admin_simplify_html_output_head_core_block_styles', isset( $_POST['kdweb_admin_simplify_html_output_head_core_block_styles'] ) );
		update_option( 'kdweb_admin_simplify_html_output_remove_menu_hash', isset( $_POST['kdweb_admin_simplify_html_output_remove_menu_hash'] ) );
		update_option( 'kdweb_admin_simplify_html_output_html5_tags', isset( $_POST['kdweb_admin_simplify_html_output_html5_tags'] ) );

		// Undeletable Content
		update_option( 'kdweb_admin_undeletable_key_pages', isset( $_POST['kdweb_admin_undeletable_key_pages'] ) );
		$undeletable_posts = isset( $_POST['kdweb_admin_undeletable_posts'] ) && is_array( $_POST['kdweb_admin_undeletable_posts'] ) ? array_map( 'intval', $_POST['kdweb_admin_undeletable_posts'] ) : array();
		update_option( 'kdweb_admin_undeletable_posts', $undeletable_posts );
		$undeletable_terms = isset( $_POST['kdweb_admin_undeletable_terms'] ) && is_array( $_POST['kdweb_admin_undeletable_terms'] ) ? array_map( 'intval', $_POST['kdweb_admin_undeletable_terms'] ) : array();
		update_option( 'kdweb_admin_undeletable_terms', $undeletable_terms );

		// Hide Blocks
		$hidden_blocks = isset( $_POST['kdweb_admin_hidden_blocks'] ) && is_array( $_POST['kdweb_admin_hidden_blocks'] ) ? array_map( 'sanitize_text_field', $_POST['kdweb_admin_hidden_blocks'] ) : array();
		update_option( 'kdweb_admin_hidden_blocks', $hidden_blocks );
		update_option( 'kdweb_admin_hide_blocks_show_all_to_admin', isset( $_POST['kdweb_admin_hide_blocks_show_all_to_admin'] ) );

		echo '<div class="updated"><p>Settings saved.</p></div>';
	}

	$enabled_features = get_option( 'kdweb_admin_enabled_features', array() );

	?>
	<div class="wrap">
		<h1>KD Web Admin Settings</h1>
		<form method="post" action="">
			<?php wp_nonce_field( 'kdweb_admin_settings_save', 'kdweb_admin_settings_nonce' ); ?>

			<h2>Logo & Icon</h2>
			<p>Set your site logo and icon. These will be used throughout the site and admin area.</p>

			<table class="form-table">
				<tr>
					<th scope="row"><label for="kdweb_site_logo">Site Logo</label></th>
					<td>
						<input type="hidden" id="kdweb_site_logo" name="kdweb_site_logo" value="<?php echo esc_attr( get_option( 'kdweb-site-logo', '' ) ); ?>" />
						<div id="kdweb_logo_preview" style="margin-bottom: 10px;">
							<?php
							$logo_id = get_option( 'kdweb-site-logo', '' );
							if ( $logo_id ) {
								$logo_url = wp_get_attachment_image_url( $logo_id, 'medium' );
								if ( $logo_url ) {
									echo '<img src="' . esc_url( $logo_url ) . '" style="max-width: 200px; height: auto; display: block;" />';
								}
							}
							?>
						</div>
						<button type="button" class="button" id="kdweb_upload_logo_button">Choose Logo</button>
						<button type="button" class="button" id="kdweb_remove_logo_button" <?php echo empty( $logo_id ) ? 'style="display:none;"' : ''; ?>>Remove Logo</button>
						<p class="description">Upload or select a logo image for your site.</p>
					</td>
				</tr>
				<tr>
					<th scope="row"><label for="kdweb_site_icon">Site Icon</label></th>
					<td>
						<input type="hidden" id="kdweb_site_icon" name="kdweb_site_icon" value="<?php echo esc_attr( get_option( 'kdweb-site-icon', '' ) ); ?>" />
						<div id="kdweb_icon_preview" style="margin-bottom: 10px;">
							<?php
							$icon_id = get_option( 'kdweb-site-icon', '' );
							if ( $icon_id ) {
								$icon_url = wp_get_attachment_image_url( $icon_id, 'thumbnail' );
								if ( $icon_url ) {
									echo '<img src="' . esc_url( $icon_url ) . '" style="max-width: 64px; height: auto; display: block;" />';
								}
							}
							?>
						</div>
						<button type="button" class="button" id="kdweb_upload_icon_button">Choose Icon</button>
						<button type="button" class="button" id="kdweb_remove_icon_button" <?php echo empty( $icon_id ) ? 'style="display:none;"' : ''; ?>>Remove Icon</button>
						<p class="description">Upload or select a site icon (favicon). Recommended size: 512x512 pixels.</p>
					</td>
				</tr>
			</table>

			<h2>Undeletable Content</h2>
			<p>Specify posts and terms that cannot be deleted from the admin.</p>

			<table class="form-table">
				<tr>
					<th scope="row">Key Pages</th>
					<td>
						<label>
							<input type="checkbox" name="kdweb_admin_undeletable_key_pages" value="1" <?php checked( get_option( 'kdweb_admin_undeletable_key_pages', false ) ); ?>>
							Make key pages undeletable
						</label>
						<p class="description">Automatically protect homepage, blog page, and WooCommerce pages (if active) from deletion.</p>
					</td>
				</tr>
				<tr>
					<th scope="row"><label for="kdweb_admin_undeletable_posts">Undeletable Posts</label></th>
					<td>
						<?php
						$undeletable_posts = get_option( 'kdweb_admin_undeletable_posts', array() );
						if ( ! is_array( $undeletable_posts ) ) {
							// Convert old comma-separated format to array
							$undeletable_posts = array_filter( array_map( 'trim', explode( ',', $undeletable_posts ) ) );
						}
						?>
						<select name="kdweb_admin_undeletable_posts[]" id="kdweb_admin_undeletable_posts" class="kdweb-post-select" multiple="multiple" style="width: 100%; max-width: 600px;">
							<?php
							foreach ( $undeletable_posts as $post_id ) {
								$post = get_post( $post_id );
								if ( $post ) {
									echo '<option value="' . esc_attr( $post_id ) . '" selected="selected">' . esc_html( $post->post_title ) . ' (ID: ' . $post_id . ', Type: ' . $post->post_type . ')</option>';
								}
							}
							?>
						</select>
						<p class="description">Search and select posts, pages, or any custom post type to protect from deletion.</p>
					</td>
				</tr>
				<tr>
					<th scope="row"><label for="kdweb_admin_undeletable_terms">Undeletable Terms</label></th>
					<td>
						<?php
						$undeletable_terms = get_option( 'kdweb_admin_undeletable_terms', array() );
						if ( ! is_array( $undeletable_terms ) ) {
							// Convert old comma-separated format to array
							$undeletable_terms = array_filter( array_map( 'trim', explode( ',', $undeletable_terms ) ) );
						}
						?>
						<select name="kdweb_admin_undeletable_terms[]" id="kdweb_admin_undeletable_terms" class="kdweb-term-select" multiple="multiple" style="width: 100%; max-width: 600px;">
							<?php
							foreach ( $undeletable_terms as $term_id ) {
								$term = get_term( $term_id );
								if ( $term && ! is_wp_error( $term ) ) {
									echo '<option value="' . esc_attr( $term_id ) . '" selected="selected">' . esc_html( $term->name ) . ' (ID: ' . $term_id . ', Taxonomy: ' . $term->taxonomy . ')</option>';
								}
							}
							?>
						</select>
						<p class="description">Search and select categories, tags, or any taxonomy terms to protect from deletion.</p>
					</td>
				</tr>
			</table>

			<?php render_hide_blocks_section(); ?>

			<h2>Features</h2>
			<p>Enable or disable features provided by the KD Web Admin plugin.</p>

			<table class="form-table">
				<tr>
					<th scope="row">Admin Role</th>
					<td>
						<label>
							<input type="checkbox" name="kdweb_admin_create_admin_role" value="1" <?php checked( in_array( 'create_admin_role', $enabled_features, true ) ); ?>>
							Create admin role between editor and administrator
						</label>
					</td>
				</tr>
				<tr>
					<th scope="row">Asset Versioning</th>
					<td>
						<label>
							<input type="checkbox" name="kdweb_admin_disable_asset_versioning" value="1" <?php checked( in_array( 'disable_asset_versioning', $enabled_features, true ) ); ?>>
							Disable CSS/JS file versioning
						</label>
						<br>
						<div style="margin: 12px 0 0 24px;">
							<label>
								<input type="checkbox" name="kdweb_admin_disable_asset_versioning_only_unset" value="1" <?php checked( get_option( 'kdweb_admin_disable_asset_versioning_only_unset', true ) ); ?>>
								Only remove automatically added versions
							</label>
						</div>
					</td>
				</tr>
				<tr>
					<th scope="row">Auto Updates</th>
					<td>
						<label>
							<input type="checkbox" name="kdweb_admin_disable_auto_updates" value="1" <?php checked( in_array( 'disable_auto_updates', $enabled_features, true ) ); ?>>
							Disable automatic updates
						</label>
						<br>
						<div style="margin: 12px 0 0 24px;">
							<label><input type="checkbox" name="kdweb_admin_disable_auto_updates_core" value="1" <?php checked( get_option( 'kdweb_admin_disable_auto_updates_core', true ) ); ?>> Core updates</label><br>
							<label><input type="checkbox" name="kdweb_admin_disable_auto_updates_minor" value="1" <?php checked( get_option( 'kdweb_admin_disable_auto_updates_minor', true ) ); ?>> Minor core updates</label><br>
							<label><input type="checkbox" name="kdweb_admin_disable_auto_updates_plugin" value="1" <?php checked( get_option( 'kdweb_admin_disable_auto_updates_plugin', true ) ); ?>> Plugin updates</label><br>
							<label><input type="checkbox" name="kdweb_admin_disable_auto_updates_theme" value="1" <?php checked( get_option( 'kdweb_admin_disable_auto_updates_theme', true ) ); ?>> Theme updates</label><br>
							<label><input type="checkbox" name="kdweb_admin_disable_auto_updates_translation" value="1" <?php checked( get_option( 'kdweb_admin_disable_auto_updates_translation', false ) ); ?>> Translation updates</label>
						</div>
					</td>
				</tr>
				<tr>
					<th scope="row">Custom CSS</th>
					<td>
						<label>
							<input type="checkbox" name="kdweb_admin_disable_custom_css" value="1" <?php checked( in_array( 'disable_custom_css', $enabled_features, true ) ); ?>>
							Disable WordPress custom CSS feature
						</label>
					</td>
				</tr>
				<tr>
					<th scope="row">Customiser</th>
					<td>
						<label>
							<input type="checkbox" name="kdweb_admin_disable_customizer" value="1" <?php checked( in_array( 'disable_customizer', $enabled_features, true ) ); ?>>
							Disable WordPress Customiser
						</label>
					</td>
				</tr>
				<tr>
					<th scope="row">Feeds</th>
					<td>
						<label>
							<input type="checkbox" name="kdweb_admin_disable_feeds" value="1" <?php checked( in_array( 'disable_feeds', $enabled_features, true ) ); ?>>
							Disable RSS/Atom feeds
						</label>
					</td>
				</tr>
				<tr>
					<th scope="row">JSON API</th>
					<td>
						<label>
							<input type="checkbox" name="kdweb_admin_disable_json_api" value="1" <?php checked( in_array( 'disable_json_api', $enabled_features, true ) ); ?>>
							Disable JSON REST API
						</label>
						<br>
						<div style="margin: 12px 0 0 24px;">
							<label>
								<input type="checkbox" name="kdweb_admin_disable_json_api_for_unauthenticated_only" value="1" <?php checked( get_option( 'kdweb_admin_disable_json_api_for_unauthenticated_only', true ) ); ?>>
								For unauthenticated users only
							</label>
						</div>
					</td>
				</tr>
				<tr>
					<th scope="row">XML-RPC</th>
					<td>
						<label>
							<input type="checkbox" name="kdweb_admin_disable_xmlprc" value="1" <?php checked( in_array( 'disable_xmlprc', $enabled_features, true ) ); ?>>
							Disable XML-RPC functionality
						</label>
					</td>
				</tr>
				<tr>
					<th scope="row">Sticky Posts</th>
					<td>
						<label>
							<input type="checkbox" name="kdweb_admin_extend_sticky_posts" value="1" <?php checked( in_array( 'extend_sticky_posts', $enabled_features, true ) ); ?>>
							Extend sticky posts to other queries
						</label>
						<br>
						<div style="margin: 12px 0 0 24px;">
							<label><input type="checkbox" name="kdweb_admin_extend_sticky_posts_author" value="1" <?php checked( get_option( 'kdweb_admin_extend_sticky_posts_author', true ) ); ?>> Author archives</label><br>
							<label><input type="checkbox" name="kdweb_admin_extend_sticky_posts_custom" value="1" <?php checked( get_option( 'kdweb_admin_extend_sticky_posts_custom', false ) ); ?>> Custom queries</label><br>
							<label><input type="checkbox" name="kdweb_admin_extend_sticky_posts_date" value="1" <?php checked( get_option( 'kdweb_admin_extend_sticky_posts_date', false ) ); ?>> Date archives</label><br>
							<label><input type="checkbox" name="kdweb_admin_extend_sticky_posts_feed" value="1" <?php checked( get_option( 'kdweb_admin_extend_sticky_posts_feed', false ) ); ?>> Feeds</label><br>
							<label><input type="checkbox" name="kdweb_admin_extend_sticky_posts_post_type" value="1" <?php checked( get_option( 'kdweb_admin_extend_sticky_posts_post_type', true ) ); ?>> Post type archives</label><br>
							<label><input type="checkbox" name="kdweb_admin_extend_sticky_posts_posts_page" value="1" <?php checked( get_option( 'kdweb_admin_extend_sticky_posts_posts_page', true ) ); ?>> Posts page</label><br>
							<label><input type="checkbox" name="kdweb_admin_extend_sticky_posts_taxonomy" value="1" <?php checked( get_option( 'kdweb_admin_extend_sticky_posts_taxonomy', true ) ); ?>> Taxonomy archives</label>
						</div>
					</td>
				</tr>
				<tr>
					<th scope="row">jQuery CDN</th>
					<td>
						<label>
							<input type="checkbox" name="kdweb_admin_load_jquery_from_cdn" value="1" <?php checked( in_array( 'load_jquery_from_cdn', $enabled_features, true ) ); ?>>
							Load jQuery from CDN
						</label>
					</td>
				</tr>
				<tr>
					<th scope="row">Login Page</th>
					<td>
						<label>
							<input type="checkbox" name="kdweb_admin_setup_customized_login_page" value="1" <?php checked( in_array( 'setup_customized_login_page', $enabled_features, true ) ); ?>>
						Customise login page
						</label>
						<br>
						<div style="margin: 12px 0 0 24px;">
							<label><input type="checkbox" name="kdweb_admin_setup_customized_login_page_logo" value="1" <?php checked( get_option( 'kdweb_admin_setup_customized_login_page_logo', false ) ); ?>> Custom logo</label><br>
							<label for="kdweb_admin_setup_customized_login_page_color" style="display: block; margin-top: 10px;">Brand colour:</label>
							<?php
							$theme_primary_color = get_theme_primary_color();
							$current_color = get_option( 'kdweb_admin_setup_customized_login_page_color', '' );
							$show_reset_button = ! empty( $theme_primary_color ) && $theme_primary_color !== $current_color;
							?>
							<div style="display: flex; align-items: center; gap: 10px;">
								<input type="text" id="kdweb_admin_setup_customized_login_page_color" name="kdweb_admin_setup_customized_login_page_color" class="kdweb-color-picker" value="<?php echo esc_attr( $current_color ); ?>" placeholder="#000000">
								<?php if ( ! empty( $theme_primary_color ) ) : ?>
									<button type="button" id="kdweb_reset_theme_color" class="button button-secondary" data-theme-color="<?php echo esc_attr( $theme_primary_color ); ?>" style="<?php echo $show_reset_button ? 'margin-top:-6px;line-height:1.5;min-height:30px;' : 'display:none;'; ?>">Reset to theme default</button>
									<input type="hidden" id="kdweb_theme_primary_color" value="<?php echo esc_attr( $theme_primary_color ); ?>">
								<?php endif; ?>
							</div>
						</div>
					</td>
				</tr>
				<tr>
					<th scope="row">Dashboard Info Widget</th>
					<td>
						<label>
							<input type="checkbox" name="kdweb_admin_setup_dashboard_info" value="1" <?php checked( in_array( 'setup_dashboard_info', $enabled_features, true ) ); ?>>
							Add system information widget to dashboard
						</label>
					</td>
				</tr>
				<tr>
					<th scope="row">Dashboard Launchpad Widget</th>
					<td>
						<label>
							<input type="checkbox" name="kdweb_admin_setup_dashboard_launchpad" value="1" <?php checked( in_array( 'setup_dashboard_launchpad', $enabled_features, true ) ); ?>>
							Add launchpad widget to dashboard
						</label>
					</td>
				</tr>
				<tr>
					<th scope="row">Simplify Admin</th>
					<td>
						<label>
							<input type="checkbox" name="kdweb_admin_simplify_admin" value="1" <?php checked( in_array( 'simplify_admin', $enabled_features, true ) ); ?>>
							Simplify WordPress admin interface
						</label>
						<p class="description">Note: Hiding and adjusting menu items does not affect administrators.</p>
						<br>
						<div style="margin: 12px 0 0 24px;">
							<label><input type="checkbox" name="kdweb_admin_simplify_admin_admin_bar" value="1" <?php checked( get_option( 'kdweb_admin_simplify_admin_admin_bar', true ) ); ?>> Simplify admin bar</label><br>
							<label><input type="checkbox" name="kdweb_admin_simplify_admin_dashboard" value="1" <?php checked( get_option( 'kdweb_admin_simplify_admin_dashboard', true ) ); ?>> Simplify dashboard</label><br>
							<label><input type="checkbox" name="kdweb_admin_simplify_admin_taxonomy_checked_ontop" value="1" <?php checked( get_option( 'kdweb_admin_simplify_admin_taxonomy_checked_ontop', true ) ); ?>> Disable taxonomy checked on top</label><br>
							<label><input type="checkbox" name="kdweb_admin_simplify_admin_image_options" value="1" <?php checked( get_option( 'kdweb_admin_simplify_admin_image_options', true ) ); ?>> Simplify image options</label><br>
							<label><input type="checkbox" name="kdweb_admin_simplify_admin_menus" value="1" <?php checked( get_option( 'kdweb_admin_simplify_admin_menus', true ) ); ?>> Adjust menus menu</label><br>
							<label><input type="checkbox" name="kdweb_admin_simplify_admin_redirection" value="1" <?php checked( get_option( 'kdweb_admin_simplify_admin_redirection', true ) ); ?>> Adjust redirection menu</label><br>
							<label><input type="checkbox" name="kdweb_admin_simplify_admin_facetwp" value="1" <?php checked( get_option( 'kdweb_admin_simplify_admin_facetwp', true ) ); ?>> Adjust FacetWP menu</label><br>
							<label><input type="checkbox" name="kdweb_admin_simplify_admin_relevanssi" value="1" <?php checked( get_option( 'kdweb_admin_simplify_admin_relevanssi', true ) ); ?>> Adjust Relevanssi menu</label><br>
							<label><input type="checkbox" name="kdweb_admin_simplify_admin_hide_themes" value="1" <?php checked( get_option( 'kdweb_admin_simplify_admin_hide_themes', true ) ); ?>> Hide themes menu</label><br>
							<label><input type="checkbox" name="kdweb_admin_simplify_admin_hide_tools" value="1" <?php checked( get_option( 'kdweb_admin_simplify_admin_hide_tools', false ) ); ?>> Hide tools menu</label><br>
							<label><input type="checkbox" name="kdweb_admin_simplify_admin_hide_settings" value="1" <?php checked( get_option( 'kdweb_admin_simplify_admin_hide_settings', false ) ); ?>> Hide settings menu</label><br>
							<label><input type="checkbox" name="kdweb_admin_simplify_admin_hide_updates" value="1" <?php checked( get_option( 'kdweb_admin_simplify_admin_hide_updates', true ) ); ?>> Hide updates menu</label><br>
							<label><input type="checkbox" name="kdweb_admin_simplify_admin_hide_acf" value="1" <?php checked( get_option( 'kdweb_admin_simplify_admin_hide_acf', false ) ); ?>> Hide ACF menu</label><br>
							<label><input type="checkbox" name="kdweb_admin_simplify_admin_hide_yoast_premium_pages" value="1" <?php checked( get_option( 'kdweb_admin_simplify_admin_hide_yoast_premium_pages', true ) ); ?>> Hide Yoast premium pages</label><br>
							<label><input type="checkbox" name="kdweb_admin_simplify_admin_fix_unqueriable_post_types" value="1" <?php checked( get_option( 'kdweb_admin_simplify_admin_fix_unqueriable_post_types', true ) ); ?>> Fix unqueriable post types</label><br>
							<label><input type="checkbox" name="kdweb_admin_simplify_admin_gf_submit_to_button" value="1" <?php checked( get_option( 'kdweb_admin_simplify_admin_gf_submit_to_button', true ) ); ?>> Change Gravity Forms submit to button</label>
						</div>
					</td>
				</tr>
				<tr>
					<th scope="row">Simplify HTML Output</th>
					<td>
						<label>
							<input type="checkbox" name="kdweb_admin_simplify_html_output" value="1" <?php checked( in_array( 'simplify_html_output', $enabled_features, true ) ); ?>>
							Simplify HTML output
						</label>
						<br>
						<div style="margin: 12px 0 0 24px;">
							<label><input type="checkbox" name="kdweb_admin_simplify_html_output_simple_mode" value="1" <?php checked( get_option( 'kdweb_admin_simplify_html_output_simple_mode', false ) ); ?>> <strong>Simple mode</strong> (enables most options below)</label><br>
							<label><input type="checkbox" name="kdweb_admin_simplify_html_output_archive_title" value="1" <?php checked( get_option( 'kdweb_admin_simplify_html_output_archive_title', true ) ); ?>> Simpler archive titles</label><br>
							<label><input type="checkbox" name="kdweb_admin_simplify_html_output_archive_link" value="1" <?php checked( get_option( 'kdweb_admin_simplify_html_output_archive_link', true ) ); ?>> Simpler archive links</label><br>
							<label><input type="checkbox" name="kdweb_admin_simplify_html_output_image_attributes" value="1" <?php checked( get_option( 'kdweb_admin_simplify_html_output_image_attributes', true ) ); ?>> Simplify image attributes</label><br>
							<label><input type="checkbox" name="kdweb_admin_simplify_html_output_comment_form" value="1" <?php checked( get_option( 'kdweb_admin_simplify_html_output_comment_form', true ) ); ?>> Simplify comment form</label><br>
							<label><input type="checkbox" name="kdweb_admin_simplify_html_output_comment_form_website" value="1" <?php checked( get_option( 'kdweb_admin_simplify_html_output_comment_form_website', true ) ); ?>> Remove comment form website field</label><br>
							<label><input type="checkbox" name="kdweb_admin_simplify_html_output_head_charset" value="1" <?php checked( get_option( 'kdweb_admin_simplify_html_output_head_charset', true ) ); ?>> Add charset meta tag</label><br>
							<label><input type="checkbox" name="kdweb_admin_simplify_html_output_head_xua_compatible" value="1" <?php checked( get_option( 'kdweb_admin_simplify_html_output_head_xua_compatible', true ) ); ?>> Add X-UA-Compatible meta tag</label><br>
							<label><input type="checkbox" name="kdweb_admin_simplify_html_output_head_viewport" value="1" <?php checked( get_option( 'kdweb_admin_simplify_html_output_head_viewport', true ) ); ?>> Add viewport meta tag</label><br>
							<label><input type="checkbox" name="kdweb_admin_simplify_html_output_head_revisit_after" value="1" <?php checked( get_option( 'kdweb_admin_simplify_html_output_head_revisit_after', false ) ); ?>> Add revisit-after meta tag</label><br>
							<label><input type="checkbox" name="kdweb_admin_simplify_html_output_head_distribution" value="1" <?php checked( get_option( 'kdweb_admin_simplify_html_output_head_distribution', false ) ); ?>> Add distribution meta tag</label><br>
							<label><input type="checkbox" name="kdweb_admin_simplify_html_output_head_wp_version" value="1" <?php checked( get_option( 'kdweb_admin_simplify_html_output_head_wp_version', true ) ); ?>> Remove WP version</label><br>
							<label><input type="checkbox" name="kdweb_admin_simplify_html_output_head_wlwmanifest_link" value="1" <?php checked( get_option( 'kdweb_admin_simplify_html_output_head_wlwmanifest_link', true ) ); ?>> Remove WLW manifest link</label><br>
							<label><input type="checkbox" name="kdweb_admin_simplify_html_output_head_rsd_link" value="1" <?php checked( get_option( 'kdweb_admin_simplify_html_output_head_rsd_link', true ) ); ?>> Remove RSD link</label><br>
							<label><input type="checkbox" name="kdweb_admin_simplify_html_output_head_shortlink" value="1" <?php checked( get_option( 'kdweb_admin_simplify_html_output_head_shortlink', true ) ); ?>> Remove shortlink</label><br>
							<label><input type="checkbox" name="kdweb_admin_simplify_html_output_head_adjacent_posts_link" value="1" <?php checked( get_option( 'kdweb_admin_simplify_html_output_head_adjacent_posts_link', true ) ); ?>> Remove adjacent posts links</label><br>
							<label><input type="checkbox" name="kdweb_admin_simplify_html_output_head_emoji" value="1" <?php checked( get_option( 'kdweb_admin_simplify_html_output_head_emoji', true ) ); ?>> Remove emoji scripts</label><br>
							<label><input type="checkbox" name="kdweb_admin_simplify_html_output_head_oembed" value="1" <?php checked( get_option( 'kdweb_admin_simplify_html_output_head_oembed', true ) ); ?>> Remove oEmbed links</label><br>
							<label><input type="checkbox" name="kdweb_admin_simplify_html_output_head_post_relational_links" value="1" <?php checked( get_option( 'kdweb_admin_simplify_html_output_head_post_relational_links', true ) ); ?>> Remove post relational links</label><br>
							<label><input type="checkbox" name="kdweb_admin_simplify_html_output_head_core_block_styles" value="1" <?php checked( get_option( 'kdweb_admin_simplify_html_output_head_core_block_styles', true ) ); ?>> Remove core block styles</label><br>
							<label><input type="checkbox" name="kdweb_admin_simplify_html_output_remove_menu_hash" value="1" <?php checked( get_option( 'kdweb_admin_simplify_html_output_remove_menu_hash', true ) ); ?>> Remove menu hash anchors</label><br>
							<label><input type="checkbox" name="kdweb_admin_simplify_html_output_html5_tags" value="1" <?php checked( get_option( 'kdweb_admin_simplify_html_output_html5_tags', true ) ); ?>> HTML5 tags</label>
						</div>
					</td>
				</tr>
			</table>

			<?php submit_button(); ?>

			<hr style="margin: 30px 0;">

			<h2>Reset Settings</h2>
			<p>Reset all settings to their default values. This action cannot be undone.</p>
			<button type="button" id="kdweb_reset_settings_button" class="button button-secondary">Reset to Defaults</button>
		</form>
	</div>
	<style>
		table.form-table label {
    		display: inline-block;
			margin-block: 5px;
		}
	</style>
	<script>
	jQuery(document).ready(function($) {
		// Feature toggles with sub-settings
		const featureToggles = {
			'kdweb_admin_disable_asset_versioning': 'kdweb_admin_disable_asset_versioning_only_unset',
			'kdweb_admin_disable_auto_updates': [
				'kdweb_admin_disable_auto_updates_core',
				'kdweb_admin_disable_auto_updates_minor',
				'kdweb_admin_disable_auto_updates_plugin',
				'kdweb_admin_disable_auto_updates_theme',
				'kdweb_admin_disable_auto_updates_translation'
			],
			'kdweb_admin_disable_json_api': 'kdweb_admin_disable_json_api_for_unauthenticated_only',
			'kdweb_admin_extend_sticky_posts': [
				'kdweb_admin_extend_sticky_posts_author',
				'kdweb_admin_extend_sticky_posts_custom',
				'kdweb_admin_extend_sticky_posts_date',
				'kdweb_admin_extend_sticky_posts_feed',
				'kdweb_admin_extend_sticky_posts_post_type',
				'kdweb_admin_extend_sticky_posts_posts_page',
				'kdweb_admin_extend_sticky_posts_taxonomy'
			],
			'kdweb_admin_setup_customized_login_page': [
				'kdweb_admin_setup_customized_login_page_logo',
				'kdweb_admin_setup_customized_login_page_color'
			],
			'kdweb_admin_simplify_admin': [
				'kdweb_admin_simplify_admin_admin_bar',
				'kdweb_admin_simplify_admin_dashboard',
				'kdweb_admin_simplify_admin_taxonomy_checked_ontop',
				'kdweb_admin_simplify_admin_image_options',
				'kdweb_admin_simplify_admin_menus',
				'kdweb_admin_simplify_admin_redirection',
				'kdweb_admin_simplify_admin_facetwp',
				'kdweb_admin_simplify_admin_relevanssi',
				'kdweb_admin_simplify_admin_hide_themes',
				'kdweb_admin_simplify_admin_hide_tools',
				'kdweb_admin_simplify_admin_hide_settings',
				'kdweb_admin_simplify_admin_hide_updates',
				'kdweb_admin_simplify_admin_hide_acf',
				'kdweb_admin_simplify_admin_hide_yoast_premium_pages',
				'kdweb_admin_simplify_admin_fix_unqueriable_post_types',
				'kdweb_admin_simplify_admin_gf_submit_to_button'
			],
			'kdweb_admin_simplify_html_output': [
				'kdweb_admin_simplify_html_output_simple_mode',
				'kdweb_admin_simplify_html_output_archive_title',
				'kdweb_admin_simplify_html_output_archive_link',
				'kdweb_admin_simplify_html_output_image_attributes',
				'kdweb_admin_simplify_html_output_comment_form',
				'kdweb_admin_simplify_html_output_comment_form_website',
				'kdweb_admin_simplify_html_output_head_charset',
				'kdweb_admin_simplify_html_output_head_xua_compatible',
				'kdweb_admin_simplify_html_output_head_viewport',
				'kdweb_admin_simplify_html_output_head_revisit_after',
				'kdweb_admin_simplify_html_output_head_distribution',
				'kdweb_admin_simplify_html_output_head_wp_version',
				'kdweb_admin_simplify_html_output_head_wlwmanifest_link',
				'kdweb_admin_simplify_html_output_head_rsd_link',
				'kdweb_admin_simplify_html_output_head_shortlink',
				'kdweb_admin_simplify_html_output_head_adjacent_posts_link',
				'kdweb_admin_simplify_html_output_head_emoji',
				'kdweb_admin_simplify_html_output_head_oembed',
				'kdweb_admin_simplify_html_output_head_post_relational_links',
				'kdweb_admin_simplify_html_output_head_core_block_styles',
				'kdweb_admin_simplify_html_output_remove_menu_hash',
				'kdweb_admin_simplify_html_output_html5_tags',
				'kdweb_admin_simplify_html_output_normalize_classnames'
			]
		};

		// Function to toggle sub-settings visibility
		function toggleSubSettings(mainCheckbox, subSettings) {
			const isChecked = $(mainCheckbox).is(':checked');
			const container = $(mainCheckbox).closest('td').find('div[style*="margin"]');

			if (isChecked) {
				container.slideDown(200);
			} else {
				container.slideUp(200);
			}
		}

		// Initialize visibility for all feature toggles
		$.each(featureToggles, function(mainId, subIds) {
			const $mainCheckbox = $('input[name="' + mainId + '"]');

			// Set initial state
			toggleSubSettings($mainCheckbox, subIds);

			// Add change handler
			$mainCheckbox.on('change', function() {
				toggleSubSettings(this, subIds);
			});
		});

		// Handle Simple Mode in Simplify HTML Output
		const $simpleMode = $('input[name="kdweb_admin_simplify_html_output_simple_mode"]');
		const simplifyHtmlOutputOptions = [
			'kdweb_admin_simplify_html_output_archive_title',
			'kdweb_admin_simplify_html_output_archive_link',
			'kdweb_admin_simplify_html_output_image_attributes',
			'kdweb_admin_simplify_html_output_comment_form',
			'kdweb_admin_simplify_html_output_comment_form_website',
			'kdweb_admin_simplify_html_output_head_charset',
			'kdweb_admin_simplify_html_output_head_xua_compatible',
			'kdweb_admin_simplify_html_output_head_viewport',
			'kdweb_admin_simplify_html_output_head_revisit_after',
			'kdweb_admin_simplify_html_output_head_distribution',
			'kdweb_admin_simplify_html_output_head_wp_version',
			'kdweb_admin_simplify_html_output_head_wlwmanifest_link',
			'kdweb_admin_simplify_html_output_head_rsd_link',
			'kdweb_admin_simplify_html_output_head_shortlink',
			'kdweb_admin_simplify_html_output_head_adjacent_posts_link',
			'kdweb_admin_simplify_html_output_head_emoji',
			'kdweb_admin_simplify_html_output_head_oembed',
			'kdweb_admin_simplify_html_output_head_post_relational_links',
			'kdweb_admin_simplify_html_output_head_core_block_styles',
			'kdweb_admin_simplify_html_output_remove_menu_hash',
			'kdweb_admin_simplify_html_output_html5_tags',
			'kdweb_admin_simplify_html_output_normalize_classnames'
		];

		function toggleSimpleModeOptions() {
			const isSimpleMode = $simpleMode.is(':checked');

			$.each(simplifyHtmlOutputOptions, function(i, optionName) {
				const $checkbox = $('input[name="' + optionName + '"]');
				if (isSimpleMode) {
					$checkbox.prop('disabled', true).closest('label').css('opacity', '0.5');
				} else {
					$checkbox.prop('disabled', false).closest('label').css('opacity', '1');
				}
			});
		}

		// Initialize simple mode state
		toggleSimpleModeOptions();

		// Add change handler for simple mode
		$simpleMode.on('change', toggleSimpleModeOptions);

		// Logo and Icon Media Uploader
		let logoFrame, iconFrame;

		// Logo uploader
		$('#kdweb_upload_logo_button').on('click', function(e) {
			e.preventDefault();

			if (logoFrame) {
				logoFrame.open();
				return;
			}

			logoFrame = wp.media({
				title: 'Select Site Logo',
				button: {
					text: 'Use as Logo'
				},
				multiple: false
			});

			logoFrame.on('select', function() {
				const attachment = logoFrame.state().get('selection').first().toJSON();
				$('#kdweb_site_logo').val(attachment.id);
				$('#kdweb_logo_preview').html('<img src="' + attachment.url + '" style="max-width: 200px; height: auto; display: block;" />');
				$('#kdweb_remove_logo_button').show();
			});

			logoFrame.open();
		});

		$('#kdweb_remove_logo_button').on('click', function(e) {
			e.preventDefault();
			$('#kdweb_site_logo').val('');
			$('#kdweb_logo_preview').html('');
			$(this).hide();
		});

		// Icon uploader
		$('#kdweb_upload_icon_button').on('click', function(e) {
			e.preventDefault();

			if (iconFrame) {
				iconFrame.open();
				return;
			}

			iconFrame = wp.media({
				title: 'Select Site Icon',
				button: {
					text: 'Use as Icon'
				},
				multiple: false
			});

			iconFrame.on('select', function() {
				const attachment = iconFrame.state().get('selection').first().toJSON();
				$('#kdweb_site_icon').val(attachment.id);
				$('#kdweb_icon_preview').html('<img src="' + attachment.url + '" style="max-width: 64px; height: auto; display: block;" />');
				$('#kdweb_remove_icon_button').show();
			});

			iconFrame.open();
		});

		$('#kdweb_remove_icon_button').on('click', function(e) {
			e.preventDefault();
			$('#kdweb_site_icon').val('');
			$('#kdweb_icon_preview').html('');
			$(this).hide();
		});

		// Initialize Select2 for posts
		$('.kdweb-post-select').select2({
			ajax: {
				url: ajaxurl,
				dataType: 'json',
				delay: 250,
				data: function(params) {
					return {
						action: 'kdweb_search_posts',
						nonce: '<?php echo esc_js( wp_create_nonce( 'kdweb_admin_search' ) ); ?>',
						q: params.term,
						page: params.page || 1
					};
				},
				processResults: function(data) {
					return {
						results: data.results,
						pagination: data.pagination
					};
				},
				cache: true
			},
			minimumInputLength: 0,
			placeholder: 'Search for posts, pages, or any post type...',
			allowClear: true,
			width: '100%'
		});

		// Initialize Select2 for terms
		$('.kdweb-term-select').select2({
			ajax: {
				url: ajaxurl,
				dataType: 'json',
				delay: 250,
				data: function(params) {
					return {
						action: 'kdweb_search_terms',
						nonce: '<?php echo esc_js( wp_create_nonce( 'kdweb_admin_search' ) ); ?>',
						q: params.term,
						page: params.page || 1
					};
				},
				processResults: function(data) {
					return {
						results: data.results,
						pagination: data.pagination
					};
				},
				cache: true
			},
			minimumInputLength: 0,
			placeholder: 'Search for categories, tags, or any taxonomy term...',
			allowClear: true,
			width: '100%'
		});

		// Initialize WordPress color picker
		$('.kdweb-color-picker').wpColorPicker({
			change: function(event, ui) {
				// Check if we need to show/hide the reset button
				const $input = $(event.target);
				if ($input.attr('id') === 'kdweb_admin_setup_customized_login_page_color') {
					const themeColor = $('#kdweb_theme_primary_color').val();
					const currentColor = ui.color.toString();

					if (themeColor && currentColor.toLowerCase() !== themeColor.toLowerCase()) {
						$('#kdweb_reset_theme_color').show();
					} else {
						$('#kdweb_reset_theme_color').hide();
					}
				}
			}
		});

		// Handle reset to theme color button
		$('#kdweb_reset_theme_color').on('click', function(e) {
			e.preventDefault();
			const themeColor = $(this).data('theme-color');
			if (themeColor) {
				$('#kdweb_admin_setup_customized_login_page_color').wpColorPicker('color', themeColor);
				$(this).hide();
			}
		});

		// Reset settings button
		$('#kdweb_reset_settings_button').on('click', function(e) {
			e.preventDefault();

			const button = $(this);
			const originalText = button.text();
			button.prop('disabled', true).text('Resetting...');

			$.ajax({
				url: ajaxurl,
				type: 'POST',
				data: {
					action: 'kdweb_reset_settings',
					nonce: '<?php echo esc_js( wp_create_nonce( 'kdweb_admin_reset' ) ); ?>'
				},
				success: function(response) {
					if (response.success) {
						// Redirect with success parameter
						const url = new URL(window.location.href);
						url.searchParams.set('settings-reset', '1');
						window.location.href = url.toString();
					} else {
						alert('Error: ' + (response.data || 'Unknown error'));
						button.prop('disabled', false).text(originalText);
					}
				},
				error: function() {
					alert('An error occurred while resetting settings.');
					button.prop('disabled', false).text(originalText);
				}
			});
		});
	});
	</script>
	<?php
}

/**
 * Add a settings link to the plugin list page.
 *
 * @param array $links An array of plugin action links.
 * @return array The modified array of action links.
 */
function add_settings_link( $links ) {
	$settings_link = '<a href="options-general.php?page=kdweb-admin-settings">Settings</a>';
	array_unshift( $links, $settings_link );
	return $links;
}
add_filter( 'plugin_action_links_kdweb-admin/kdweb-admin.php', 'KDWeb\Plugin\Admin\add_settings_link' );

/**
 * Register hide blocks settings
 */
function register_hide_blocks_settings() {
	register_setting( 'kdweb_admin_settings', 'kdweb_admin_hidden_blocks', array(
		'type'              => 'array',
		'sanitize_callback' => function( $value ) {
			return is_array( $value ) ? array_map( 'sanitize_text_field', $value ) : array();
		},
		'default'           => array(),
	) );
	register_setting( 'kdweb_admin_settings', 'kdweb_admin_hide_blocks_show_all_to_admin', array(
		'type'              => 'boolean',
		'sanitize_callback' => 'rest_sanitize_boolean',
		'default'           => false,
	) );
}
add_action( 'admin_init', 'KDWeb\Plugin\Admin\register_hide_blocks_settings' );

/**
 * Render hide blocks section in settings page
 */
function render_hide_blocks_section() {
	require_once __DIR__ . '/hide-blocks.php';

	$all_blocks      = get_all_block_types();
	$hidden_blocks   = get_option( 'kdweb_admin_hidden_blocks', array() );
	$show_all_admin  = get_option( 'kdweb_admin_hide_blocks_show_all_to_admin', false );
	$default_settings = get_default_settings();
	$default_hidden_blocks = $default_settings['kdweb_admin_hidden_blocks'];
	?>
	<h2><?php esc_html_e( 'Hide Blocks', 'kdweb-admin' ); ?></h2>
	<table class="form-table">
		<tr>
			<th scope="row"><?php esc_html_e( 'Show All to Administrators', 'kdweb-admin' ); ?></th>
			<td>
				<label>
					<input type="checkbox" name="kdweb_admin_hide_blocks_show_all_to_admin" value="1" <?php checked( $show_all_admin, true ); ?> />
					<?php esc_html_e( 'Show all blocks to administrators even if hidden', 'kdweb-admin' ); ?>
				</label>
			</td>
		</tr>
		<tr>
			<th scope="row"><?php esc_html_e( 'Hidden Blocks', 'kdweb-admin' ); ?></th>
			<td>
				<?php if ( empty( $all_blocks ) ) : ?>
					<p class="description">
						<?php esc_html_e( 'No blocks cached yet. Open the block editor on any post/page to collect available blocks.', 'kdweb-admin' ); ?>
					</p>
				<?php else : ?>
					<p style="margin-bottom: 10px;">
						<a href="#" id="kdweb_toggle_all_blocks" class="button button-secondary"><?php esc_html_e( 'Toggle All', 'kdweb-admin' ); ?></a>
						<a href="#" id="kdweb_set_to_defaults_blocks" class="button button-secondary" style="display: none;"><?php esc_html_e( 'Set to Defaults', 'kdweb-admin' ); ?></a>
					</p>
					<fieldset>
						<legend class="screen-reader-text"><?php esc_html_e( 'Select blocks to hide', 'kdweb-admin' ); ?></legend>
						<div id="kdweb_blocks_list" style="max-height: 400px; overflow-y: auto; border: 1px solid #ddd; padding: 10px; background: #fff;">
							<?php foreach ( $all_blocks as $block_name => $block_title ) : ?>
								<label style="display: block; margin-bottom: 8px; padding: 4px;">
									<input type="checkbox" name="kdweb_admin_hidden_blocks[]" value="<?php echo esc_attr( $block_name ); ?>" <?php checked( in_array( $block_name, (array) $hidden_blocks, true ) ); ?> class="kdweb-block-checkbox" />
									<code style="background: #f0f0f1; padding: 2px 6px; border-radius: 3px;"><?php echo esc_html( $block_name ); ?></code>
									- <?php echo esc_html( $block_title ); ?>
								</label>
							<?php endforeach; ?>
						</div>
						<p class="description" style="margin-top: 10px;">
							<?php esc_html_e( 'Select blocks to hide from the block editor for non-administrators.', 'kdweb-admin' ); ?>
						</p>
					</fieldset>
				<?php endif; ?>
			</td>
		</tr>
	</table>
	<script>
	jQuery(document).ready(function($) {
		const defaultHiddenBlocks = <?php echo wp_json_encode( $default_hidden_blocks ); ?>;
		const $checkboxes = $('.kdweb-block-checkbox');
		const $setToDefaultsBtn = $('#kdweb_set_to_defaults_blocks');

		// Function to check if current state matches defaults
		function checkIfMatchesDefaults() {
			// Get all registered block names (blocks that have checkboxes)
			const registeredBlocks = [];
			$checkboxes.each(function() {
				registeredBlocks.push($(this).val());
			});

			// Get currently checked blocks
			const currentHidden = [];
			$checkboxes.filter(':checked').each(function() {
				currentHidden.push($(this).val());
			});

			// Filter defaults to only include blocks that are actually registered
			const defaultsForRegistered = defaultHiddenBlocks.filter(function(blockName) {
				return registeredBlocks.indexOf(blockName) !== -1;
			});

			// Sort both arrays to compare
			const currentSorted = currentHidden.slice().sort();
			const defaultSorted = defaultsForRegistered.slice().sort();

			// Check if arrays are equal
			const isEqual = currentSorted.length === defaultSorted.length &&
				currentSorted.every((val, index) => val === defaultSorted[index]);

			// Show/hide button based on whether state matches defaults
			if (isEqual) {
				$setToDefaultsBtn.hide();
			} else {
				$setToDefaultsBtn.show();
			}
		}

		// Check initial state
		checkIfMatchesDefaults();

		// Check state when checkboxes change
		$checkboxes.on('change', function() {
			checkIfMatchesDefaults();
		});

		// Toggle all functionality
		$('#kdweb_toggle_all_blocks').on('click', function(e) {
			e.preventDefault();
			const allChecked = $checkboxes.length === $checkboxes.filter(':checked').length;
			$checkboxes.prop('checked', !allChecked);
			checkIfMatchesDefaults();
		});

		// Set to defaults functionality
		$setToDefaultsBtn.on('click', function(e) {
			e.preventDefault();
			// Uncheck all first
			$checkboxes.prop('checked', false);
			// Check only the defaults
			$checkboxes.each(function() {
				if (defaultHiddenBlocks.indexOf($(this).val()) !== -1) {
					$(this).prop('checked', true);
				}
			});
			checkIfMatchesDefaults();
		});
	});
	</script>
	<?php
}
