<?php
/**
 * Hide blocks from block editor dynamically
 *
 * @package KDWeb\Plugin\Admin
 */

namespace KDWeb\Plugin\Admin;

use WP_REST_Request;

if ( ! defined( 'ABSPATH' ) ) { exit; }

/**
 * Register REST API endpoint for caching JS block types
 */
add_action( 'rest_api_init', function() {
	register_rest_route( 'kdweb/v1', '/cache-block-types', array(
		'methods'             => 'POST',
		'callback'            => 'KDWeb\Plugin\Admin\cache_js_block_types',
		'permission_callback' => function() {
			return current_user_can( 'manage_options' );
		},
	) );
} );

/**
 * Cache JS block types from the editor
 *
 * @param WP_REST_Request $request The request object.
 * @return array Response with cached block names.
 */
function cache_js_block_types( WP_REST_Request $request ) {
	$js_blocks = $request->get_param( 'blocks' );
	update_option( 'kdweb_admin_cached_js_blocks', $js_blocks );
	return rest_ensure_response( array( 'cached' => true, 'blocks' => $js_blocks ) );
}

/**
 * Get PHP registered block types
 *
 * @return array Array of block types with name and title.
 */
function get_php_block_types() {
	$registry    = \WP_Block_Type_Registry::get_instance();
	$block_types = array();

	foreach ( $registry->get_all_registered() as $block_name => $block_type ) {
		$block_types[] = array(
			'name'  => $block_name,
			'title' => $block_type->title,
		);
	}

	update_option( 'kdweb_admin_cached_php_blocks', $block_types );
	return $block_types;
}

/**
 * Get all block types (PHP and JS combined)
 *
 * @return array Array of all block types.
 */
function get_all_block_types() {
	$php_blocks = get_option( 'kdweb_admin_cached_php_blocks', array() );
	$js_blocks  = get_option( 'kdweb_admin_cached_js_blocks', array() );

	// If caches are empty, refresh them
	if ( empty( $php_blocks ) ) {
		$php_blocks = get_php_block_types();
	}

	$all_blocks = array();

	// Add PHP blocks
	foreach ( $php_blocks as $block ) {
		$all_blocks[ $block['name'] ] = $block['title'];
	}

	// Add JS blocks (now includes title)
	if ( is_array( $js_blocks ) ) {
		foreach ( $js_blocks as $block ) {
			if ( ! isset( $all_blocks[ $block['name'] ] ) ) {
				$all_blocks[ $block['name'] ] = $block['title'];
			}
		}
	}

	ksort( $all_blocks );
	return $all_blocks;
}

/**
 * Filter allowed block types in the editor
 *
 * @param array|bool $allowed_block_types Array of allowed block types or true/false.
 * @return array|bool Modified array of allowed block types.
 */
function filter_allowed_block_types( $allowed_block_types ) {
	// Check if administrators should see all blocks
	$show_all_to_admin = get_option( 'kdweb_admin_hide_blocks_show_all_to_admin', false );
	if ( $show_all_to_admin && current_user_can( 'manage_options' ) ) {
		return $allowed_block_types;
	}

	$hidden_blocks = get_option( 'kdweb_admin_hidden_blocks', array() );

	if ( empty( $hidden_blocks ) ) {
		return $allowed_block_types;
	}

	// Get all available blocks
	$all_blocks = array_keys( get_all_block_types() );

	// Remove hidden blocks
	$allowed = array_diff( $all_blocks, $hidden_blocks );

	return array_values( $allowed );
}
add_filter( 'allowed_block_types_all', 'KDWeb\Plugin\Admin\filter_allowed_block_types', 10, 1 );

/**
 * Enqueue script to collect JS blocks in the editor
 */
function enqueue_block_collector_script() {
	if ( ! is_admin() ) {
		return;
	}

	$screen = get_current_screen();
	if ( $screen && $screen->is_block_editor() ) {
		wp_enqueue_script(
			'kdweb-collect-block-types',
			plugin_dir_url( dirname( __FILE__ ) ) . 'assets/collect-block-types.js',
			array( 'wp-blocks', 'wp-data', 'wp-api-fetch' ),
			'1.0.0',
			true
		);
	}
}
add_action( 'admin_enqueue_scripts', 'KDWeb\Plugin\Admin\enqueue_block_collector_script' );
