<?php
/**
 * Extend sticky posts feature to work on other queries
 *
 * by default the feature only works on main posts archive
 * this feature also changes how sticky posts work by simply reordering posts results to show sticky ones first
 * the default behaviour is to put sticky posts on top of the first page as extra posts in the list (on top of posts_per_page)
 * the setting 'posts_page' overrides default behaviour for the the main posts archive as well
 * for more granular control, you can use the 'ignore_sticky_posts' query var in your queries, see wp docs
 *
 * @package KDWeb\Plugin\Admin
 */

namespace KDWeb\Plugin\Admin;

if ( ! defined( 'ABSPATH' ) ) { exit; }

/**
 * Change order for
 *
 * @param string $orderby Orderby clause.
 * @param WP_Query $wp_query The WP_Query instance.
 * @return string
 */
function sticky_posts_orderby( $orderby, $wp_query ) {
	global $wpdb;
	$sticky_posts = get_option( 'sticky_posts' );
	if ( ! is_sticky_posts_enabled_for_query( $wp_query ) ) {
		return $orderby;
	}
	$sticky_posts = implode( ',', $sticky_posts );
	$orderby = "FIELD( {$wpdb->posts}.ID, $sticky_posts ) DESC, $orderby";
	return $orderby;
}

/**
 * Add sticky class to posts where applicable
 *
 * @param string|array $classes The classes.
 * @param string|array $class The class as a string.
 * @param int          $id The post ID.
 * @return array
 */
function sticky_posts_add_class( $classes, $class, $post_id ) {
	global $wp_query;
	if( is_sticky( $post_id ) && ! in_array( 'sticky', $classes ) && is_sticky_posts_enabled_for_query( $wp_query ) ) {
		$classes[] = 'sticky';
	}
	return $classes;
}

/**
 * Whether the extended sticky posts feature is enabled for the current query
 *
 * @param WP_Query $wp_query The WP_Query instance.
 * @return boolean
 */
function is_sticky_posts_enabled_for_query( $wp_query ) {
	$sticky_posts = get_option( 'sticky_posts' );
	if ( ! is_array( $sticky_posts ) || empty( $sticky_posts ) || $wp_query->query_vars['ignore_sticky_posts'] ) {
		return false;
	}

	// Basic exclusions
	if ( $wp_query->is_singular() || $wp_query->is_search() || $wp_query->is_comment_feed() || $wp_query->is_robots() || $wp_query->is_404() ) {
		return false;
	}

	// Author archives
	if ( $wp_query->is_author() && ! get_option( 'kdweb_admin_extend_sticky_posts_author', true ) ) {
		return false;
	}

	// Custom queries (not main query)
	if ( ! $wp_query->is_main_query() && ! get_option( 'kdweb_admin_extend_sticky_posts_custom', false ) ) {
		return false;
	}

	// Date archives
	if ( ( $wp_query->is_date() || $wp_query->is_year() || $wp_query->is_month() || $wp_query->is_time() ) &&
		 ! get_option( 'kdweb_admin_extend_sticky_posts_date', false ) ) {
		return false;
	}

	// Feeds
	if ( $wp_query->is_feed() && ! get_option( 'kdweb_admin_extend_sticky_posts_feed', false ) ) {
		return false;
	}

	// Post type archives
	if ( $wp_query->is_post_type_archive() && ! get_option( 'kdweb_admin_extend_sticky_posts_post_type', true ) ) {
		return false;
	}

	// Posts page
	if ( ! $wp_query->is_home() && ! get_option( 'kdweb_admin_extend_sticky_posts_posts_page', true ) ) {
		return false;
	}

	// Taxonomy archives
	if ( ( $wp_query->is_category() || $wp_query->is_tag() || $wp_query->is_tax() ) &&
		 ! get_option( 'kdweb_admin_extend_sticky_posts_taxonomy', true ) ) {
		return false;
	}

	return true;
}
