<?php
/**
 * Create an admin role between editor and administrator levels
 *
 * @package KDWeb\Plugin\Admin
 */

namespace KDWeb\Plugin\Admin;

if ( ! defined( 'ABSPATH' ) ) { exit; }

/**
 * Register the admin role
 *
 * @return void
 */
function register_admin_role() {
	$administrator = get_role( 'administrator' );
	$administrator->add_cap( 'kdweb_manage_code_slots' );
	$administrator->add_cap( 'kdweb_access_admin' );
    add_role(
        'admin',
        __( 'Admin' ),
        apply_filters(
            'kdweb_admin_role_capabilities',
            array(
                'activate_plugins' => false,
				'add_users' => true,
                'create_users' => true,
				'customize' => false,
                'delete_pages' => true,
                'delete_others_pages' => true,
                'delete_private_pages' => true,
                'delete_published_pages' => true,
				'create_patterns' => false,
                'delete_posts' => true,
                'delete_others_posts' => true,
                'delete_private_posts' => true,
                'delete_published_posts' => true,
				'delete_patterns' => false,
                'delete_plugins' => false,
                'delete_themes' => false,
                'delete_users' => true,
                'edit_dashboard' => true,
                'edit_files' => false,
                'edit_pages' => true,
                'edit_others_pages' => true,
                'edit_private_pages' => true,
                'edit_published_pages' => true,
				'edit_patterns' => false,
                'edit_posts' => true,
                'edit_others_posts' => true,
                'edit_private_posts' => true,
                'edit_published_posts' => true,
                'edit_plugins' => false,
                'edit_theme_options' => true,
                'edit_themes' => false,
                'edit_users' => true,
                'export' => false,
                'import' => false,
                'install_plugins' => false,
                'install_themes' => false,
                'list_users' => true,
                'manage_categories' => true,
                'manage_links' => false,
                'manage_options' => false,
				'manage_privacy_options' => true,
                'moderate_comments' => true,
                'promote_users' => true,
                'publish_posts' => true,
                'publish_pages' => true,
                'read' => true,
				'read_patterns' => true,
                'read_private_pages' => true,
                'read_private_posts' => true,
                'remove_users' => true,
                'switch_themes' => false,
                'update_core' => false,
                'update_plugins' => false,
                'update_themes' => false,
                'unfiltered_html' => true,
                'unfiltered_upload' => false,
                'upload_files' => true,

				'gform_full_access' => false,
				'gravityforms_api_settings' => false,
				'gravityforms_create_form' => true,
				'gravityforms_delete_entries' => true,
    			'gravityforms_delete_forms' => true,
				'gravityforms_edit_entries' => true,
				'gravityforms_edit_entry_notes' => true,
				'gravityforms_edit_forms' => true,
				'gravityforms_edit_settings' => false,
				'gravityforms_export_entries' => true,
				'gravityforms_logging' => false,
				'gravityforms_preview_forms' => false,
				'gravityforms_system_status' => false,
				'gravityforms_view_addons' => false,
				'gravityforms_view_entries' => true,
				'gravityforms_view_entry_notes' => true,
				'gravityforms_view_settings' => false,
				'gravityforms_view_updates' => false,
				'gravityforms_uninstall' => false,

				'hicpo_load_script_css' => true,
				'hicpo_update_menu_order' => true,
				'hicpo_update_menu_order_tags' => true,

				'assign_product_terms' => true,
				'assign_shop_coupon_terms' => true,
				'assign_shop_order_terms' => true,
				'create_customers' => true,
				'delete_others_products' => true,
				'delete_others_shop_coupons' => true,
				'delete_others_shop_orders' => true,
				'delete_private_products' => true,
				'delete_private_shop_coupons' => true,
				'delete_private_shop_orders' => true,
				'delete_product' => true,
				'delete_product_terms' => true,
				'delete_products' => true,
				'delete_published_products' => true,
				'delete_published_shop_coupons' => true,
				'delete_published_shop_orders' => true,
				'delete_shop_coupon' => true,
				'delete_shop_coupon_terms' => true,
				'delete_shop_coupons' => true,
				'delete_shop_order' => true,
				'delete_shop_order_terms' => true,
				'delete_shop_orders' => true,
				'edit_others_products' => true,
				'edit_others_shop_coupons' => true,
				'edit_others_shop_orders' => true,
				'edit_private_products' => true,
				'edit_private_shop_coupons' => true,
				'edit_private_shop_orders' => true,
				'edit_product' => true,
				'edit_product_terms' => true,
				'edit_products' => true,
				'edit_published_products' => true,
				'edit_published_shop_coupons' => true,
				'edit_published_shop_orders' => true,
				'edit_shop_coupon' => true,
				'edit_shop_coupon_terms' => true,
				'edit_shop_coupons' => true,
				'edit_shop_order' => true,
				'edit_shop_order_terms' => true,
				'edit_shop_orders' => true,
				'manage_product_terms' => true,
				'manage_shop_coupon_terms' => true,
				'manage_shop_order_terms' => true,
				'manage_woocommerce' => true,
				'publish_products' => true,
				'publish_shop_coupons' => true,
				'publish_shop_orders' => true,
				'read_others_products' => true,
				'read_others_shop_coupons' => true,
				'read_others_shop_orders' => true,
				'read_private_products' => true,
				'read_private_shop_coupons' => true,
				'read_private_shop_orders' => true,
				'read_product' => true,
				'read_product_terms' => true,
				'read_products' => true,
				'read_published_products' => true,
				'read_published_shop_coupons' => true,
				'read_published_shop_orders' => true,
				'read_shop_coupon' => true,
				'read_shop_coupon_terms' => true,
				'read_shop_coupons' => true,
				'read_shop_order' => true,
				'read_shop_order_terms' => true,
				'read_shop_orders' => true,
				'view_woocommerce_reports' => true,

				'wpseo_bulk_edit' => false,
				'wpseo_edit_advanced_metadata' => true,
				'wpseo_manage_options' => true,

				'view_site_health_checks' => false,

				'kdweb_access_admin' => true,
				'kdweb_manage_code_slots' => true,
            )
        )
    );
}

/**
 * Get a list of roles the current user can edit
 *
 * @return array
 */
function admin_role_allowed_roles() {
	global $wp_roles;
	if ( current_user_can( 'administrator' ) ) {
		return array_keys( $wp_roles->roles );
	}
	if ( ! current_user_can( 'admin' ) ) {
		return array();
	}
	return array_filter(
		array_keys( $wp_roles->roles ),
		fn( $role ) => 'administrator' !== $role
	);
}

/**
 * Overwrite capabilites so that user with admin role cannot edit or delte administrators
 *
 * @param array $caps The capabilites related to the mapping.
 * @param array $cap The current capability.
 * @param int   $user_id The acting user's id.
 * @param array $args Additional parameters for the action.
 * @return array
 */
function admin_role_map_meta_cap( $caps, $cap, $user_id, $args ) {
	if ( 'edit_user' !== $cap && 'delete_user' !== $cap || empty( $args ) || ! isset( $args[0] ) ) {
		return $caps;
	}
	$user   = get_userdata( $user_id );
	$edited = get_userdata( $args[0] );
	if ( ! $user || ! $edited || $user->ID === $edited->ID ) {
		return $caps;
	}
	$allowed = admin_role_allowed_roles();
	if ( array_diff( $edited->roles, $allowed ) ) {
		$caps[] = 'not_allowed';
	}
	return $caps;
}

/**
 * Remove roles that are not allowed for admins to edit
 *
 * @param array $roles The array of roles.
 * @return array
 */
function admin_role_limit_editable_roles( $roles ) {
	if ( ! current_user_can( 'admin' ) ) {
		return $roles;
	}
	$allowed = admin_role_allowed_roles();
	foreach ( array_keys( $roles ) as $role ) {
		if ( ! in_array( $role, $allowed, true ) ) {
			unset( $roles[ $role ] );
		}
	}
	return $roles;
}

/**
 * Add criteria to exclude users with 'Administrator' role from users list for lower level users
 *
 * @param wpdb $user_query The database user query object.
 * @return wpdb
 */
function admin_role_hide_administrators_for_lower_level_users( $user_query ) {
	global $wpdb;
	if ( current_user_can( 'administrator' ) ) {
		return;
	}
	$query = $wpdb->prepare(
		"SELECT user_id FROM {$wpdb->usermeta} WHERE user_id!=%d AND meta_key=%s AND meta_value LIKE %s",
		array(
			get_current_user_id(),
			$wpdb->prefix . 'capabilities',
			'%"administrator"%',
		)
	);
	$ids = $wpdb->get_col( $query );
	if ( is_array( $ids ) && count( $ids ) > 0 ) {
		$ids = implode( ',', $ids );
		$user_query->query_where .= " AND ( $wpdb->users.ID NOT IN ( $ids ) )";
	}
}

/**
 * Exclude administrators filter from the users list for lower level users
 *
 * @param array $views The array of views.
 * @return array
 */
function admin_role_exclude_administrator_filter_for_lower_users( $views ) {
	if ( ! isset( $views['administrator'] ) || current_user_can( 'administrator' ) ) {
		return $views;
	}
	unset( $views['administrator'] );
	return $views;
}

/**
 * Allow admin roles to translate strings in polylang
 *
 * @return void
 */
function admin_role_allow_polylang() {
    if ( current_user_can( 'manage_options' ) || ! function_exists( 'PLL' ) ) {
        return;
    }
    add_menu_page(
        __( 'Strings translations', 'polylang' ),
        __( 'Languages', 'polylang' ),
        'admin',
        'mlang_strings',
        array(
            \PLL(), /* phpcs:ignore */
            'languages_page'
        ),
        'dashicons-translation',
        50
    );
}

/**
 * Enable 'Admin' role to access to the Facet WP plugin
 */
function admin_role_allow_access() {
	return 'kdweb_access_admin';
}
